#pragma once

#include <yxiva/core/operation_result.h>
#include "expression_parser.h"
#include "definition_translator.h"
#include <yxiva/core/json.h>

namespace yxiva { namespace filter { namespace parser {

class logic_expressions_filter_translator
{
public:
    bool operator()(const json_value& data)
    {
        auto& json_vars = data["vars"];
        if (!validate_vars_json(json_vars)) return false;

        auto& json_rules = data["rules"];
        if (!validate_rules_json(json_rules)) return false;

        rules_t rules;
        vars_t variables;
        std::set<string> used_vars_names;
        action default_action = action::send_bright;
        if (!translate_rules(rules, default_action, used_vars_names, variables, json_rules))
            return false;

        if (!translate_definitions(variables, used_vars_names, json_vars)) return false;

        result = filter_set(std::move(rules), std::move(variables), default_action);
        return true;
    }

    const string get_error() const
    {
        return error_reason;
    }

    void move_result_to(filter_set& dst)
    {
        dst = std::move(result);
    }

private:
    /** Rules are represented as json array of json objects each of which represent separate rule.
     * Rule's position in that array defines it's priority (rule at position 0 has the highest
     * priority).
     *
     * Rule format:
     * { "if": "<expression>", "do" : "<action>" } --- conditional rule
     * or
     * { "do" : "<action>" } --- unconditional rule, it's equilent to conditional rule with "if":""
     * After the first unconditional rule is met in the rules array all further rules can be ignored
     * ignored, so the translator skips checking their semantic correctness.
     */
    bool translate_rules(
        rules_t& rules,
        action& default_action,
        std::set<string>& used_vars_names,
        vars_t& vars,
        const json_value& json_rules)
    {
        rules.reserve(json_rules.size());
        for (auto&& json_rule : json_rules.array_items())
        {
            if (!validate_rule(json_rule)) return false;

            auto&& json_do = json_rule["do"];
            action rule_action = filter::action_from_string(json_do.to_string());

            auto&& json_if = json_rule["if"];
            // unconditional rule
            if (is_null(json_if))
            {
                default_action = rule_action;
                break; // no need to go further after first unconditional rule was met
            }
            else if (json_if.type() == json_type::tstring)
            {
                auto json_if_str = json_if.to_string();
                if (json_if_str.empty())
                {
                    default_action = rule_action;
                    break; // no need to go further after first unconditional rule was met
                }

                // conditional rule
                rule new_rule;
                new_rule.action = rule_action;
                if (!parse_rule_if(new_rule.expr, used_vars_names, json_if_str)) return false;
                rules.push_back(new_rule);
            }
            else
            {

                // rule of one operand that is declared right in rule
                definition_translator translate;
                if (!translate(json_if, ""))
                {
                    error_reason = translate.get_error();
                    return false;
                }
                var_t var;
                translate.move_result_to(var);
                add_single_variable_rule(rules, vars, std::move(var.second), rule_action);
            }
        }
        return true;
    }

    bool parse_rule_if(expression& dst, std::set<string>& vars_names, const string& json_if)
    {
        expression_parser parser;
        if (!parser(json_if))
        {
            error_reason = parser.get_error();
            return false;
        }
        parser.move_result_to(dst, vars_names);
        return true;
    }

    bool translate_definitions(
        vars_t& variables,
        const std::set<string>& used_vars_names,
        const json_value& json_vars)
    {
        for (const string& var_name : used_vars_names)
        {
            auto& json_var = json_vars[var_name];
            definition_translator translate;
            if (!translate(json_var, var_name))
            {
                error_reason = translate.get_error();
                return false;
            }
            var_t var;
            translate.move_result_to(var);
            variables.emplace(std::move(var));
        }
        return true;
    }

    // Validators

    bool validate_vars_json(const json_value& json_vars)
    {
        if (json_vars.type() != json_type::tnull && json_vars.type() != json_type::tobject)
        {
            error_reason = "invalid vars node";
            return false;
        }
        return true;
    }

    bool validate_rules_json(const json_value& json_rules)
    {
        if (json_rules.type() != json_type::tarray)
        {
            error_reason = "invalid rules node type";
            return false;
        }
        if (json_rules.empty())
        {
            error_reason = "empty rules node";
            return false;
        }
        return true;
    }

    bool validate_rule(const json_value& json_rule)
    {
        if (json_rule.type() != json_type::tobject)
        {
            error_reason = "invalid rule type";
            return false;
        }
        auto& json_do = json_rule["do"];
        if (json_do.type() != json_type::tstring ||
            action_from_string(json_do.to_string()) == action::unknown)
        {
            error_reason = "invalid action in rule";
            return false;
        }

        auto& json_if = json_rule["if"];
        if (json_if.type() != json_type::tnull && json_if.type() != json_type::tstring &&
            json_if.type() != json_type::tobject)
        {
            error_reason = "invalid expression type";
            return false;
        }
        return true;
    }

private:
    filter_set result;
    string error_reason;
};

inline operation::result decode_filter_v2(filter_set& dst, const json_value& data) noexcept
{
    logic_expressions_filter_translator translate;
    if (!translate(data)) return translate.get_error();

    translate.move_result_to(dst);
    return operation::success;
}

}}}
