#pragma once

#include "decoders.h"
#include <yxiva/core/filter.h>
#include <yxiva/core/json.h>

namespace yxiva { namespace filter { namespace parser {

template <typename Condition>
class basic_definition_translator
{
    using condition_t = Condition;
    using cond_t = typename condition_t::cond_t;

public:
    bool operator()(const json_value& json_var, const std::string& var_name)
    {
        if (!validate_root_json_var(json_var, var_name)) return false;

        auto iitem = json_var.members_begin();
        condition_t cond;
        if (!translate_condition(cond, string(iitem.key()), *iitem))
        {
            return false;
        }
        if (!validate_condition(cond)) return false;

        definition = { var_name, std::move(cond) };
        return true;
    }

    const string get_error() const
    {
        return error_reason;
    }

    void move_result_to(::yxiva::filter::basic_var_t<condition_t>& dest)
    {
        dest = std::move(definition);
    }

    void move_result_to(condition_t& dest)
    {
        dest = std::move(definition.second);
    }

private:
    bool translate_condition(
        condition_t& cond,
        string key,
        const json_value& json,
        bool expect_only_predicate = false)
    {
        bool translate_predicate =
            expect_only_predicate || condition_from_string<cond_t>(key) != cond_t::unknown;
        if (translate_predicate)
        {
            if (!validate_predicate_json(key, json)) return false;
        }
        else
        {
            if (!validate_condition_json(key, json)) return false;
        }

        if (translate_predicate)
        {
            if (!translate_predicate_and_values(cond, key, json)) return false;
        }
        else
        {
            cond.key = std::move(key);
            auto iitem = json.members_begin();
            if (!translate_condition(cond, string(iitem.key()), *iitem, true)) return false;
        }

        return true;
    }

    bool translate_predicate_and_values(
        condition_t& cond,
        const string& key,
        const json_value& json)
    {
        cond.type = condition_from_string<cond_t>(key);
        if (cond.type == cond_t::unknown)
        {
            error_reason = "unknown condition type";
            return false;
        }

        auto values_decoded = decode_values(cond.value, json);
        if (!values_decoded)
        {
            error_reason = values_decoded.error_reason;
            return false;
        }
        return true;
    }

    // Validators

    bool validate_root_json_var(const json_value& json_var, const std::string& var_name)
    {
        if (is_null(json_var))
        {
            error_reason = "missing definition for '" + var_name + "'";
            return false;
        }
        if (json_var.type() != json_type::tobject || json_var.size() != 1)
        {
            error_reason = "invalid definition for '" + var_name + "'";
            return false;
        }
        return true;
    }

    bool validate_predicate_json(const string& key, const json_value& json)
    {
        if (condition_from_string<cond_t>(key) == cond_t::unknown)
        {
            error_reason = "invalid predicate key";
            return false;
        }
        if (json.empty())
        {
            error_reason = "empty value";
            return false;
        }
        if (json.type() != json_type::tarray)
        {
            error_reason = "invalid value type";
            return false;
        }
        return true;
    }

    bool validate_condition_json(const string& key, const json_value& json)
    {
        if (json.empty())
        {
            error_reason = "empty condition json, field expected";
            return false;
        }
        if (key.empty())
        {
            error_reason = "empty condition key";
            return false;
        }
        if (json.type() != json_type::tobject)
        {
            error_reason = "invalid predicate";
            return false;
        }
        return true;
    }

    bool validate_condition(const condition_t& cond)
    {
        bool uses_key = condition_uses_key(cond.type);
        bool empty = cond.key.empty();
        if (empty && uses_key)
        {
            error_reason = "key not specified";
            return false;
        }
        if (!empty && !uses_key)
        {
            error_reason = "key specified for non-field predicate";
            return false;
        }
        return true;
    }

private:
    basic_var_t<condition_t> definition;
    string error_reason;
};

using definition_translator = basic_definition_translator<condition>;

}}}
