#pragma once

#include <string>
#include <vector>
#include <set>

namespace yxiva { namespace filter { namespace parser {

using std::string;

class expression_parser
{
public:
    /** Parse logic expression of some variables.
     * Supported operations:
     *    binary: & --- AND
     *    unary: ! --- NOT
     * Braces are not supported
     */
    bool operator()(const string& expr_str)
    {
        const int EXPECT_VAR = 1;
        const int EXPECT_UNARY_OP = 2;
        const int EXPECT_BINARY_OP = 4;
        const int EXPECT_END = 8;

        int expected = EXPECT_VAR | EXPECT_UNARY_OP | EXPECT_END;
        auto curr = expr_str.begin();
        auto end = expr_str.end();

        while (curr != end)
        {
            curr = std::find_if_not(curr, end, isspace);
            if (curr == end) break;

            const char ch = *curr;
            if (isalpha(ch))
            {
                if (!(expected & EXPECT_VAR))
                {
                    error_reason = "unexcpected variable start";
                    return false;
                }
                auto name_end = std::find_if_not(curr, end, isalnum);
                expr.operands.emplace_back(curr, name_end);
                vars_names.emplace(curr, name_end);
                expected = EXPECT_BINARY_OP | EXPECT_END;
                curr = name_end;
                continue;
            }

            switch (ch)
            {
            case '!':
                if (!(expected & EXPECT_UNARY_OP))
                {
                    error_reason = "unexcpected unary operation";
                    return false;
                }
                expr.operators.push_back(OP_NOT);
                expected = EXPECT_VAR;
                break;
            case '&':
                if (!(expected & EXPECT_BINARY_OP))
                {
                    error_reason = "unexcpected binary operation";
                    return false;
                }
                expr.operators.push_back(OP_AND);
                expected = EXPECT_UNARY_OP | EXPECT_VAR;
                break;
            default:
                error_reason = string("unexpected symbol in expression '") + ch + "'";
                return false;
            }
            ++curr;
        }
        if (!(expected & EXPECT_END))
        {
            error_reason = "bad expression ending";
            return false;
        }
        return true;
    }

    const string get_error() const
    {
        return error_reason;
    }

    void move_result_to(expression& dest_expr, std::set<string>& dest_vars_names)
    {
        dest_expr = std::move(expr);
        std::move(
            vars_names.begin(),
            vars_names.end(),
            std::inserter(dest_vars_names, dest_vars_names.end()));
    }

private:
    expression expr;
    std::set<string> vars_names;
    string error_reason;
};

}}}
