#include <yxiva/core/platforms.h>
#include <yplatform/encoding/base64.h>
#include <ctime>

namespace yxiva { namespace apns {

certificate_type certificate_type_from_env(app_environment env)
{
    if (env == app_environment::production) return certificate_type::production;
    if (env == app_environment::development) return certificate_type::development;
    throw std::runtime_error("invalid environment for apns cert: " + get_environment_name(env));
}

certificate_type resolve_certificate_type(const string& name)
{
    static const std::map<string, certificate_type> CERT_TYPE_BY_NAME{
        { "development", certificate_type::development },
        { "production", certificate_type::production }
    };
    auto it = CERT_TYPE_BY_NAME.find(name);
    if (it == CERT_TYPE_BY_NAME.end())
    {
        throw std::runtime_error("can't resolve certificate type " + name);
    }
    return it->second;
}

const string& get_certificate_type_name(certificate_type type)
{
    static const std::map<certificate_type, string> CERTIFICATE_TYPE_NAMES{
        { certificate_type::development, "development" },
        { certificate_type::production, "production" }
    };
    auto it = CERTIFICATE_TYPE_NAMES.find(type);
    if (it == CERTIFICATE_TYPE_NAMES.end())
    {
        throw std::runtime_error("unknown certificate type");
    }
    return it->second;
}

p8_token::p8_token(const json_value& json)
    : key(yplatform::base64_decode_str(json["key"].to_string_view()))
    , key_id(json["key_id"].to_string_view())
    , issuer_key(json["issuer_key"].to_string_view())
    , topic(json["topic"].to_string_view())
    , type(json["type"].to_string_view())
{
}

p8_token::p8_token(const string& secret)
    : p8_token{ [&secret]() {
        json_value json;
        json.parse(secret);
        return json;
    }() }
{
}

p8_token::p8_token(
    const string& key,
    const string& key_id,
    const string& issuer_key,
    const string& topic,
    const string& type)
    : key(key), key_id(key_id), issuer_key(issuer_key), topic(topic), type(type)
{
}

json_value p8_token::to_json() const
{
    json_value json;
    json["key"] = yplatform::base64_encode_str(key);
    json["key_id"] = key_id;
    json["issuer_key"] = issuer_key;
    json["topic"] = topic;
    json["type"] = type;
    return json;
}

string p8_token::to_string() const
{
    return to_json().stringify();
}

namespace {

auto parse_pem(const string& cert)
{
    X509* x509_cert_ptr;
    auto x509_cert_parse_result = x509::parse_pem(cert.data(), cert.size(), x509_cert_ptr);
    if (!x509_cert_parse_result)
    {
        throw std::runtime_error(x509_cert_parse_result.error_reason);
    }

    EVP_PKEY* private_key_ptr;
    auto private_key_parse_result =
        x509::parse_private_key(cert.data(), cert.size(), private_key_ptr);
    if (!private_key_parse_result)
    {
        X509_free(x509_cert_ptr);
        throw std::runtime_error(private_key_parse_result.error_reason);
    }

    return x509::certificate(x509_cert_ptr, private_key_ptr);
}

}

namespace pem_certificate {

string from_p12(const string& p12_cert, const string& password)
{
    x509::certificate_chain certs;
    auto parse_result = x509::parse_p12(p12_cert.data(), p12_cert.size(), password, certs);
    if (!parse_result)
    {
        throw std::runtime_error(parse_result.error_reason);
    }

    for (auto&& cert : certs)
    {
        int64_t ts;
        auto valid = cert.expiration_ts(ts);
        if (!valid)
        {
            throw std::runtime_error(valid.error_reason);
        }
    }

    string pem;
    auto write_result = x509::write_pem(certs, pem);
    if (!write_result)
    {
        throw std::runtime_error(write_result.error_reason);
    }

    return pem;
}

bool contains_pem(const string& cert)
{
    return x509::contains_pem(cert.data(), cert.size());
}

void validate(const string& cert)
{
    auto x509_cert = parse_pem(cert);
    auto res = x509_cert.up_to_date();
    if (!res)
    {
        throw std::runtime_error(res.error_reason);
    }
}

int64_t get_expiration_time(const string& cert)
{
    auto x509_cert = parse_pem(cert);

    int64_t ts = 0;
    auto res = x509_cert.expiration_ts(ts);
    if (!res)
    {
        throw std::runtime_error(res.error_reason);
    }
    return ts;
}

string get_topic(const string& cert)
{
    auto x509_cert = parse_pem(cert);

    // TODO: extract topic from extensions, if available
    string topic;
    auto res = x509_cert.subject_text(x509::attribute_type::user_id, topic);
    if (!res)
    {
        throw std::runtime_error(res.error_reason);
    }

    return topic;
}

certificate_type get_type(const string& cert)
{
    auto x509_cert = parse_pem(cert);

    string common_name;
    auto res = x509_cert.subject_text(x509::attribute_type::common_name, common_name);
    if (!res)
    {
        throw std::runtime_error(res.error_reason);
    }

    // Spaces are relevant, because the word Development may occur
    // in bundle ID, which is also a part of common name, something
    // like "Apple IOS Push Services: com.foo.bar.Development.baz".
    static const std::string development = " Development ";
    static const std::string production = " Production ";

    // It is not documented, but so far we've seen the word
    // "Development" in common names of certificates issued
    // only for Apple Development push services, not for
    // production. But in XCode the production certificate
    // is called "Development and Production", so check for
    // absense of "Production" in common name, just in case.
    // NB: not all production certificates have the word "Production"
    // in their common names, but all development certificates
    // have the word "Development".
    auto type = certificate_type::production;
    if (common_name.find(development) != string::npos &&
        common_name.find(production) == string::npos)
    {
        type = certificate_type::development;
    }

    return type;
}

}

}}
