#include <yxiva/core/repacker.h>
#include <yxiva/core/platforms.h>
#include <yplatform/util/sstream.h>

namespace yxiva {

namespace {
static const std::string WILDCARD_RULE = "*";

bool is_xiva_value_request(const std::string& name)
{
    static const std::string XIVA_PARAM_PREFIX = "::xiva::";
    return name.size() > XIVA_PARAM_PREFIX.length() &&
        name.compare(0, XIVA_PARAM_PREFIX.length(), XIVA_PARAM_PREFIX) == 0;
}

bool is_known_xiva_value(const std::string& name)
{
    static const std::set<std::string> KNONW_XIVA_PARAMS = { "::xiva::transit_id",
                                                             "::xiva::push_token" };
    return KNONW_XIVA_PARAMS.count(name) > 0;
}
}

string apply_repack_features(
    const packet& packet,
    const push_subscription_params& subscription,
    const string& raw_payload);

void apply_repack_features(
    const packet& packet,
    const push_subscription_params& subscription,
    json_value& payload);

void inject_notification_id(json_value& payload, const string& transit_id, const string& platform);

string appmetrica_notification_id(const packet& packet);

operation::result custom_repacker::from_json_string(
    const string& raw_repacking_rules,
    custom_repacker& custom_repacker)
{
    json_value unpacked_js;
    auto parse_result = json_parse(unpacked_js, raw_repacking_rules);
    if (!parse_result) return parse_result;
    return custom_repacker::from_json(std::move(unpacked_js), custom_repacker);
}

operation::result custom_repacker::from_json(
    json_value js_repacking_rules,
    custom_repacker& custom_repacker)
{
    custom_repacker.push_service_params = std::move(js_repacking_rules);

    json_value repack_payload = custom_repacker.push_service_params["repack_payload"];
    custom_repacker.push_service_params.remove_member("repack_payload");

    if (repack_payload.type() == json_type::tnull)
    {
        custom_repacker.pass_payload_as_is = true;
        return operation::success;
    }

    if (repack_payload.type() != json_type::tarray)
    {
        return "invalid repack_payload type";
    }

    custom_repacker.pass_payload_as_is = false;
    if (repack_payload.empty()) return operation::success;

    for (auto irule = repack_payload.array_begin(); irule != repack_payload.array_end(); ++irule)
    {
        auto&& rule = *irule;
        if (rule.type() == json_type::tstring)
        {
            string rule_string = rule.to_string();
            if (rule_string == WILDCARD_RULE)
            {
                custom_repacker.include_source_payload = true;
            }
            else
            {
                // include field without renaming
                custom_repacker.payload.emplace_back(rule_string);
            }
        }
        else if (rule.type() == json_type::tobject)
        {
            // include field with renaming or fill xiva value

            if (rule.size() != 1) return "invalid repack_payload rule";

            const auto& rename_rule = rule.members_begin();
            if ((*rename_rule).type() != json_type::tstring) return "invalid repack_payload rule";

            std::string src_key = (*rename_rule).to_string();
            if (is_xiva_value_request(src_key) && !is_known_xiva_value(src_key))
                return "unknown xiva value request";

            custom_repacker.payload.emplace_back(string(rename_rule.key()), std::move(src_key));
        }
        else
        {
            return "invalid repack_payload rule type";
        }
    }
    custom_repacker.pass_payload_as_is =
        custom_repacker.include_source_payload && custom_repacker.payload.empty();

    return operation::success;
}

operation::result custom_repacker::validate(const json_value& source_payload) const
{
    if (payload.empty()) return operation::success;

    bool is_src_payload_empty_str = source_payload.is_string() && source_payload.size() == 0;

    if (!is_src_payload_empty_str && source_payload.type() != json_type::tobject)
        return "repack_payload for raw strings not supported";

    for (auto& rule : payload)
    {
        if (is_xiva_value_request(rule.source_key))
        {
            if (!is_known_xiva_value(rule.source_key)) return "unkonwn xiva parameter request";
        }
        else
        {
            if (is_src_payload_empty_str || !source_payload.has_member(rule.source_key))
                return "unknown key request in repack_payload";
        }
    }

    return operation::success;
}

std::tuple<operation::error, push_requests_queue> custom_repacker::repack(
    const packet& packet,
    const push_subscription_params& subscription)
{
    push_requests_queue result;
    push_request req;

    bool collapse_id_specified = false;
    for (auto option = push_service_params.members_begin();
         option != push_service_params.members_end();
         ++option)
    {
        auto key = option.key();
        if ((subscription.platform == platform::FCM && key == "notification") ||
            (subscription.platform == platform::HMS && key == "notification") ||
            (subscription.platform == platform::APNS && key == "aps") ||
            ((subscription.platform == platform::MPNS || subscription.platform == platform::WNS) &&
             key == "toast"))
        {
            req.set_bright();
        }
        if (subscription.platform == platform::APNS && key == "collapse-id")
        {
            collapse_id_specified = true;
        }
        req.add_push_service_parameter(string(key), json_write(*option));
    }
    if (!collapse_id_specified && subscription.repack_features.inject_collapse_id)
    {
        req.add_push_service_parameter(
            "collapse-id", apns::collapse_id_from_transit_id(packet.message.transit_id));
    }
    // raw_data shouldn't be parsed if no repack payload rules specified
    if (payload.size())
    {
        json_value raw_data;
        if (packet.message.raw_data.size())
        {
            // empty raw_data treats the same as empty Json object
            auto raw_data_parse_result = json_parse(raw_data, packet.message.raw_data);
            if (!raw_data_parse_result) return { "raw_data is not JSON", {} };
        }

        json_value dest_payload;
        if (include_source_payload)
        {
            dest_payload = raw_data;
        }
        for (auto& rule : payload)
        {
            if (is_xiva_value_request(rule.source_key))
            {
                if (rule.source_key == "::xiva::transit_id")
                {
                    dest_payload[rule.target_key] = packet.message.transit_id;
                }
                else if (rule.source_key == "::xiva::push_token")
                {
                    dest_payload[rule.target_key] = subscription.push_token;
                }
                else
                {
                    return { "unkonwn xiva parameter request", {} };
                }
            }
            else
            {
                if (include_source_payload && rule.source_key != rule.target_key)
                    dest_payload.remove_member(rule.source_key);
                if (raw_data.has_member(rule.source_key))
                    dest_payload[rule.target_key] = raw_data[rule.source_key];
            }
        }
        apply_repack_features(packet, subscription, dest_payload);
        req.set_payload(dest_payload.stringify());
    }
    else if (pass_payload_as_is)
    {
        req.set_payload(apply_repack_features(packet, subscription, packet.message.raw_data));
    } // else payload remains empty: ""

    result.emplace_back(std::move(req));
    return { {}, result };
}

boost::optional<string> find_custom_repacker_raw(
    const string& platform,
    const std::map<string, string>& raw_repackers)
{
    for (auto& pair : raw_repackers)
    {
        if (platform::resolve_alias(platform).name == platform::resolve_alias(pair.first).name)
        {
            return pair.second;
        }
    }
    static const string OTHER_PS_TYPE("other");
    auto iraw_repacker = raw_repackers.find(OTHER_PS_TYPE);
    return iraw_repacker == raw_repackers.end() ? boost::optional<string>{} : iraw_repacker->second;
}

std::tuple<operation::error, push_requests_queue> apns_queue_repacker::repack(
    const packet& packet,
    const push_subscription_params& sub)
{
    push_requests_queue result;

    push_request req;
    bool collapse_id_specified = false;
    if (auto params_repacker =
            find_custom_repacker_raw(sub.platform, packet.message.repacking_rules))
    {
        json_value repacker_json;
        if (auto error = repacker_json.parse(*params_repacker))
        {
            return { *error, {} };
        }
        for (auto rule_it = repacker_json.members_begin(); rule_it != repacker_json.members_end();
             ++rule_it)
        {
            if (rule_it.key() == "collapse-id")
            {
                collapse_id_specified = true;
            }
            req.add_push_service_parameter(string(rule_it.key()), json_write(*rule_it));
        }
    }
    if (!collapse_id_specified && sub.repack_features.inject_collapse_id)
    {
        req.add_push_service_parameter(
            "collapse-id", apns::collapse_id_from_transit_id(packet.message.transit_id));
    }
    json_value payload;
    if (auto error = payload.parse(packet.message.raw_data))
    {
        return { *error, {} };
    }
    if (payload.has_member("xiva"))
    {
        if (packet.message.local_id)
        {
            payload["xiva"]["pos"] = packet.message.local_id;
            payload["xiva"]["device"] =
                sub.device.size() > 10 ? sub.device.substr(sub.device.size() - 10) : sub.device;
        }
    }
    req.set_payload(json_write(payload));
    result.push_back(std::move(req));
    return { {}, result };
}

namespace hacks {
class mail_repacker_android : public repacker
{
public:
    std::tuple<operation::error, push_requests_queue> repack(
        const packet& packet,
        const push_subscription_params& subscription) override
    {
        push_requests_queue result;
        if (is_operation_unsupported(packet.message.operation))
        {
            // Operation will be ignored if no requests generated.
            return { {}, result };
        }
        string payload;
        if (auto generated = gen_payload_android(packet, subscription, payload))
        {
            // Never bright
            push_request custom_request(std::move(payload));
            if (packet.message.operation == "insert")
            {
                custom_request.add_push_service_parameter("priority", "\"high\"");
            }
            result.push_back(custom_request);
        }
        return { {}, result };
    }

private:
    operation::result gen_payload_android(
        const packet& packet,
        const push_subscription_params& subscription,
        string& payload)
    {
        json_value src_json;
        auto parse_result = json_parse(src_json, packet.message.raw_data);
        if (!parse_result)
        {
            return parse_result;
        }

        static const std::vector<field> BASE_FIELDS = { { "uname" },
                                                        { "operation" },
                                                        { "lcn" },
                                                        { "fid" },
                                                        { "tab" },
                                                        { "fids", "fids_str" },
                                                        { "mids", "mids_str" },
                                                        { "m_lids", "m_lids_str" },
                                                        { "tid", "threadId" },
                                                        { "new_messages", "newCount" },
                                                        { "status" },
                                                        { "all_labels", "all_labels_str" },
                                                        { "counters", "countersNew" } };

        json_value base_json;
        base_json["uid"] = packet.message.uid;
        base_json["transit-id"] = packet.message.transit_id;
        transform(base_json, src_json, BASE_FIELDS);
        apply_repack_features(packet, subscription, base_json);
        payload = json_write(base_json);
        return {};
    }

    bool is_operation_unsupported(const string& operation)
    {
        static const std::set<string> unsupported_operations = { "update labels",
                                                                 "delete mails",
                                                                 "reset fresh" };
        return unsupported_operations.count(operation);
    }
};

class mail_repacker_ios : public repacker
{
public:
    std::tuple<operation::error, push_requests_queue> repack(
        const packet& packet,
        const push_subscription_params& subscription) override
    {
        push_requests_queue result;
        if (is_operation_unsupported(packet.message.operation))
        {
            // Operation will be ignored if no requests generated.
            return { {}, result };
        }
        string bright, silent;
        if (auto generated = gen_payload_apns(packet, subscription, bright, silent))
        {
            if (bright.size())
            {
                push_request bright_request(std::move(bright), true);
                bright_request.add_push_service_parameter("collapse-id", packet.message.transit_id);
                result.push_back(bright_request);
            }
            if (silent.size())
            {
                push_request silent_request(std::move(silent));
                silent_request.add_custom_parameter("ttl", "0");
                result.push_back(silent_request);
            }
        }
        return { {}, result };
    }

private:
    operation::result gen_payload_apns(
        const packet& packet,
        const push_subscription_params& subscription,
        string& bright,
        string& silent)
    {
        const auto& message = packet.message;
        json_value src_json;
        auto parse_result = json_parse(src_json, message.raw_data);
        if (!parse_result)
        {
            return parse_result;
        }

        static const std::vector<field> BASE_FIELDS = { { "u", "uname" },
                                                        { "operation" },
                                                        { "lcn" },
                                                        { "fid" },
                                                        { "tab" },
                                                        { "m", "mid" },
                                                        { "m_lids", "m_lids_str" },
                                                        { "tid", "threadId" },
                                                        { "new_messages", "newCount" },
                                                        { "counters", "countersNew" },
                                                        { "avatar", "avatarUrl" } };

        json_value base_json;
        base_json["z"] = message.uid;
        base_json["transit-id"] = message.transit_id;
        base_json["local-id"] = message.local_id;
        transform(base_json, src_json, BASE_FIELDS);
        string fid_type;
        auto fid_type_it = message.data.find("fid_type");
        if (fid_type_it != message.data.end()) fid_type = fid_type_it->second;

        apply_repack_features(packet, subscription, base_json);

        bool marked_read =
            message.data.count("hdr_status") && message.data.find("hdr_status")->second == "RO";

        if (packet.bright && message.operation == "insert" && !marked_read &&
            allowed_fid_type_for_insert(fid_type))
        {
            json_value bright_json = base_json;
            bright_json["aps"]["sound"] = get_sound(subscription.extra_data);
            bright_json["aps"]["category"] = "M";
            int ios_version = get_ios_version(subscription.client);
            auto&& alert = bright_json["aps"]["alert"];
            auto&& loc_args = src_json["loc-args"];
            if (ios_version < 10 || !loc_args.is_array() || loc_args.size() != 3)
            {
                alert["loc-key"] = "p";
                static const std::vector<field> BRIGHT_FIELDS = { { "loc-args" } };
                transform(alert, src_json, BRIGHT_FIELDS);
            }
            else
            {
                alert["title"] = loc_args[0UL];
                alert["subtitle"] = loc_args[1UL];
                alert["body"] = loc_args[2UL];
            }
            // enables a notification extension
            bright_json["aps"]["mutable-content"] = 1;
            bright = json_write(bright_json);
        }
        else if (message.operation == "update labels")
        {
            static const std::vector<field> LABELS_FIELDS = { { "fids" }, { "mids" } };
            transform(base_json, src_json, LABELS_FIELDS);
        }
        // silent push must contain both of this elements
        base_json["aps"]["content-available"] = 1;
        base_json["aps"]["sound"] = "";
        silent = json_write(base_json);
        return {};
    }

    bool allowed_fid_type_for_insert(const string& fid_type)
    {
        // https://a.yandex-team.ru/arc/trunk/arcadia/mail/macs/include/macs/data/symbols.h?rev=4378508
        static const std::set<string> ignored_fid_types = {
            "2", "3", "4", "5", "6", "7", "8", "10"
        };
        return !ignored_fid_types.count(fid_type);
    }

    bool is_operation_unsupported(const string& operation)
    {
        static const std::set<string> unsupported_operations = { "delete mails", "reset fresh" };
        return unsupported_operations.count(operation);
    }

    int get_ios_version(const string& client)
    {
        size_t start = client.rfind("OS_");
        if (start == string::npos)
        {
            return 0;
        }
        start += 3;
        try
        {
            return std::stoi(client.substr(start, client.find("_", start) - start));
        }
        catch (...)
        {
            return 0;
        }
    }

    std::string get_sound(const std::string& extra)
    {
        static const std::string default_sound = "p.caf";

        json_value src_json;
        if (!json_parse(src_json, extra))
        {
            // TODO Purge if mobile app is not being released.
            static const std::string prefix = "sound.";
            using boost::algorithm::starts_with;
            if (starts_with(extra, prefix))
            {
                return extra.substr(prefix.size());
            }
            return default_sound;
        }
        string ret = json_get(src_json, "sound", default_sound);
        return ret;
    }
};
class disk_repacker : public repacker
{
public:
    std::tuple<operation::error, push_requests_queue> repack(
        const packet& packet,
        const push_subscription_params& subscription) override
    {
        static const std::set<string> localized_events = { "share_invite_new",
                                                           "space_is_low",
                                                           "space_is_full" };
        push_requests_queue result;
        json_value raw_data;
        auto raw_data_parse_result = json_parse(raw_data, packet.message.raw_data);
        if (!raw_data_parse_result) return { "raw_data is not JSON", {} };

        json_value payload(json_type::tobject);
        json_value aps(json_type::tobject);

        if (localized_events.count(packet.message.operation))
        {
            auto localized_msg = packet.message.data.find("localized_msg");
            if (localized_msg == packet.message.data.end())
            {
                return { {}, result };
            }
            payload["t"] = packet.message.operation;
            if (android_platform(subscription.platform))
            {
                payload["m"] = localized_msg->second;
            }
            else if (subscription.platform == platform::APNS)
            {
                aps["alert"] = localized_msg->second;
            }
        }
        else
        {
            auto&& root = raw_data["root"];
            if (!root.is_object()) return { "unexpected 'root' type", {} };
            if (!root.has_member("tag")) return { "missing 'tag' in root", {} };

            payload["t"] = root["tag"];
            if (root["tag"] == "photoslice_updated")
            {
                auto&& parameters = root["parameters"];
                if (!parameters.is_object()) return { "unexpected 'parameters' type", {} };
                if (!parameters.has_member("photoslice_id") ||
                    !parameters.has_member("current_revision"))
                    return { "missing photoslice paramters", {} };
                payload["p_id"] = parameters["photoslice_id"];
                payload["p_r"] = parameters["current_revision"];
            }
        }

        if (android_platform(subscription.platform))
        {
            payload["uid"] = packet.uid;
        }
        // aps.content_available is set in mobile if needed

        push_request req(json_write(payload));
        if (aps.size())
        {
            req.add_push_service_parameter("aps", json_write(aps));
            // Only ios notifications are bright.
            req.set_bright();
        }
        result.emplace_back(std::move(req));

        return { {}, result };
    }
};
}

std::tuple<operation::error, push_requests_queue> bypass_repacker::repack(
    const packet& packet,
    const push_subscription_params& subscription)
{
    push_requests_queue result;
    push_request req;
    if (subscription.repack_features.inject_transit_id)
    {
        req.set_payload(apply_repack_features(packet, subscription, packet.message.raw_data));
    }
    else
    {
        req.set_payload(packet.message.raw_data);
    }
    if (subscription.repack_features.inject_collapse_id)
    {
        req.add_push_service_parameter(
            "collapse-id", apns::collapse_id_from_transit_id(packet.message.transit_id));
    }
    result.emplace_back(std::move(req));
    return { {}, result };
}

std::tuple<operation::error, repacker_ptr> create_repacker(
    const packet& packet,
    const push_subscription_params& subscription)
{
    const auto& platform = subscription.platform;

    if (platform::ALL_SUPPORTED.count(platform) == 0)
    {
        return { "unsupported platform", {} };
    }

    if (platform == platform::APNS_QUEUE)
    {
        return { {}, std::make_shared<apns_queue_repacker>() };
    }

    if (auto custom = find_custom_repacker_raw(platform, packet.message.repacking_rules))
    {
        auto result = std::make_shared<custom_repacker>();
        auto init_result = custom_repacker::from_json_string(*custom, *result);
        if (!init_result) return { init_result.error_reason, {} };
        return { {}, result };
    }

    if (packet.service == "mail")
    {
        return { {},
                 platform == platform::APNS ?
                     repacker_ptr(std::make_shared<hacks::mail_repacker_ios>()) :
                     repacker_ptr(std::make_shared<hacks::mail_repacker_android>()) };
    }

    if (packet.service == "disk-json" || packet.service == "disk-user-events")
    {
        return { {}, std::make_shared<hacks::disk_repacker>() };
    }

    return { {}, std::make_shared<bypass_repacker>() };
}

operation::result repack_message_if_needed(
    const packet& packet,
    const push_subscription_params& subscription,
    push_requests_queue& requests)
{
    auto [error, repacker] = create_repacker(packet, subscription);
    if (error) return string(error);

    auto [repack_error, result] = repacker->repack(packet, subscription);
    if (repack_error) return string(repack_error);

    requests = result;
    return operation::success;
}

operation::result repack_message_if_needed(
    const packet& packet,
    const push_subscription_params& subscription,
    push_requests_queue& requests,
    push_request_cache& cache)
{
    // disk repack is dependent on the subscription and can't be cached
    bool use_cache =
        packet.message.service != "disk-json" && packet.message.service != "disk-user-events";
    // check for specific repacking rules preventing caching
    if (use_cache)
    {
        if (auto raw_repacker =
                find_custom_repacker_raw(subscription.platform, packet.message.repacking_rules))
        {
            use_cache = (raw_repacker->find("::xiva::push_token") == string::npos);
        }
    }
    if (use_cache)
    {
        if (auto res = cache.try_get(subscription, packet))
        {
            requests = res->second; // push requests queue
            return res->first;      // result
        }
    }
    auto repack_result = repack_message_if_needed(packet, subscription, requests);
    if (use_cache) cache.store(subscription, packet, repack_result, requests);
    return repack_result;
}

string webpush_repacker::repack(const packet& packet, const repack_features& features)
{
    json_value payload;
    if (features != repack_features{} && json_parse(payload, packet.message.raw_data))
    {
        if (features.inject_transit_id)
        {
            payload["xiva"]["transit_id"] = packet.message.transit_id;
        }
        if (features.inject_collapse_id)
        {
            payload["hacks_ts"] = std::to_string(std::time(nullptr));
        }
        return json_write(payload);
    }
    return packet.message.raw_data;
}

string apply_repack_features(
    const packet& packet,
    const push_subscription_params& subscription,
    const string& raw_payload)
{
    json_value payload;
    if (json_parse(payload, raw_payload))
    {
        apply_repack_features(packet, subscription, payload);
        return json_write(payload);
    } // ignore non-object payloads
    return raw_payload;
}

void apply_repack_features(
    const packet& packet,
    const push_subscription_params& subscription,
    json_value& payload)
{
    if (subscription.repack_features.inject_transit_id)
    {
        if (subscription.repack_features.transit_id_appmetrica_format)
        {
            inject_notification_id(
                payload, appmetrica_notification_id(packet), subscription.platform);
        }
        else
        {
            inject_notification_id(payload, packet.message.transit_id, subscription.platform);
        }
    }
}

void inject_notification_id(json_value& payload, const string& transit_id, const string& platform)
{
    if (payload.has_member("yamp") && payload["yamp"].type() == json_type::tobject)
    {
        auto key = platform == platform::APNS ? "i" : "a";
        payload["yamp"][key] = transit_id;
    }
    else
    {
        payload["xiva"]["transit_id"] = transit_id;
    }
}

string appmetrica_notification_id(const packet& packet)
{
    string complex_transit_id;
    yplatform::sstream ss(complex_transit_id, 64);
    ss << "xt=" << yplatform::url_encode(packet.message.transit_id)
       << "&xe=" << yplatform::url_encode(packet.message.operation);
    return complex_transit_id;
}

}
