#include <yxiva/core/shards/shards.h>

namespace yxiva { namespace shard_config {

namespace {
string zero_pad(id_t id, std::size_t total_len)
{
    string s = std::to_string(id);
    total_len = std::max(total_len, s.size());
    s.insert(s.begin(), total_len - s.size(), '0');
    return s;
}
}

boost::optional<const shard&> shard_from_id(const shards& shards, id_t id)
{
    auto it =
        std::find_if(shards.begin(), shards.end(), [id](const shard& s) { return s.id == id; });
    return it == shards.end() ? boost::optional<const shard&>{} :
                                boost::optional<const shard&>{ *it };
}

boost::optional<const shard&> shard_from_gid(const shards& shards, gid_t gid)
{
    auto it = std::find_if(shards.begin(), shards.end(), [gid](const shard& s) {
        return s.start_gid <= gid && gid <= s.end_gid;
    });
    return it == shards.end() ? boost::optional<const shard&>{} :
                                boost::optional<const shard&>{ *it };
}

boost::optional<const db_instance&> master_from_gid(const shards& shards, gid_t gid)
{
    auto shard = shard_from_gid(shards, gid);
    return shard ? shard->master : boost::optional<const db_instance&>{};
}

boost::optional<const db_instance&> random_replica_from_gid(const shards& shards, gid_t gid)
{
    auto shard = shard_from_gid(shards, gid);
    if (!shard || shard->replicas.empty())
    {
        return boost::optional<const db_instance&>{};
    }
    auto random_index = rand() % shard->replicas.size();
    return shard->replicas[random_index];
}

string shard::describe() const
{
    string range_str = "[" + std::to_string(start_gid) + ", " + std::to_string(end_gid) + "]";
    return zero_pad(id, 2) + ": " + range_str;
}

json_value shard::to_json() const
{
    json_value shard_node;
    shard_node["id"] = id;
    shard_node["start_gid"] = start_gid;
    shard_node["end_gid"] = end_gid;
    shard_node["master"] = master.conninfo;
    auto&& replicas_node = shard_node["replicas"];
    replicas_node.set_array();
    for (auto&& replica : replicas)
    {
        replicas_node.push_back(replica.conninfo);
    }
    return shard_node;
}

bool operator==(const db_instance& lhs, const db_instance& rhs)
{
    return lhs.conninfo == rhs.conninfo;
}

bool operator!=(const db_instance& lhs, const db_instance& rhs)
{
    return !operator==(lhs, rhs);
}

bool operator==(const shard& lhs, const shard& rhs)
{
    return lhs.id == rhs.id && lhs.start_gid == rhs.start_gid && lhs.end_gid == rhs.end_gid &&
        lhs.master == rhs.master && lhs.replicas == rhs.replicas;
}

bool operator!=(const shard& lhs, const shard& rhs)
{
    return !operator==(lhs, rhs);
}

json_value to_json(const shards& shards)
{
    json_value result;
    auto&& shards_array = result["shards"];
    shards_array.set_array();
    for (auto&& shard : shards)
    {
        shards_array.push_back(shard.to_json());
    }

    return result;
}

}}
