#include <yxiva/core/x509.h>
#include "asn1/time.h"
#include <openssl/pem.h>
#include <openssl/bio.h>
#include <openssl/pkcs12.h>

namespace yxiva { namespace x509 {

namespace {

#if OPENSSL_VERSION_NUMBER < 0x10100000L

const ASN1_TIME* X509_get0_notBefore(const X509* x)
{
    return x->cert_info->validity->notBefore;
}

const ASN1_TIME* X509_get0_notAfter(const X509* x)
{
    return x->cert_info->validity->notAfter;
}

#endif

string get_last_openssl_error()
{
    unsigned long code = ERR_peek_last_error();
    if (!code)
    {
        return string();
    }

    const char* reason = ERR_reason_error_string(code);
    if (!reason)
    {
        return "unknown openssl error";
    }

    return reason;
}

string nid_description(int nid)
{
    const char* long_name_ptr = OBJ_nid2ln(nid);
    string long_name(long_name_ptr ? long_name_ptr : "unknown");
    return std::to_string(nid) + " (" + long_name + ")";
}

operation::result create_readonly_bio(const char* data, std::size_t unchecked_size, BIO*& bio)
{
    if (unchecked_size > std::numeric_limits<int>::max())
    {
        return operation::result(
            "certificate size " + std::to_string(unchecked_size) + " is too big");
    }

    int size = static_cast<int>(unchecked_size);

    bio = BIO_new_mem_buf(const_cast<char*>(data), size);
    if (bio == nullptr)
    {
        return operation::result(
            "failed to create openssl BIO from memory: " + get_last_openssl_error());
    }
    return operation::success;
}

void free_readonly_bio(BIO* bio)
{
    BIO_free(bio);
}

}

certificate::certificate(X509* cert) : certificate(cert, nullptr)
{
}

certificate::certificate(X509* cert, EVP_PKEY* private_key)
    : cert_(cert, X509_free), pkey_(private_key, EVP_PKEY_free)
{
    assert(cert != nullptr);
}

operation::result certificate::issuer_text(attribute_type attr, string& text)
{
    return read_x509_text(X509_get_issuer_name(cert_.get()), static_cast<int>(attr), text);
}

operation::result certificate::subject_text(attribute_type attr, string& text)
{
    return read_x509_text(X509_get_subject_name(cert_.get()), static_cast<int>(attr), text);
}

operation::result certificate::expiration_ts(int64_t& ts)
{
    auto time = X509_get0_notAfter(cert_.get());
    return time ? asn1::convert_asn1time(time, ts) : "can't get expiration time";
}

operation::result certificate::up_to_date()
{
    time_t now_utc = time(nullptr);

    const ASN1_TIME* not_before = X509_get0_notBefore(cert_.get());
    const ASN1_TIME* not_after = X509_get0_notAfter(cert_.get());

    // XXX: after 2050 this will no longer work, because
    // X509_cmp_time cannot handle GeneralizedTime in openssl 1.0.1f
    // quoting RFC 2459:
    // CAs conforming to this profile MUST always encode certificate
    // validity dates through the year 2049 as UTCTime; certificate validity
    // dates in 2050 or later MUST be encoded as GeneralizedTime.
    // end quote
    // openssl>=1.0.2 can handle GeneralizedTime
    int cmp_result = X509_cmp_time(not_before, &now_utc);
    if (!cmp_result)
    {
        return operation::result("failed to read not_before: " + get_last_openssl_error());
    }

    if (cmp_result > 0)
    {
        return operation::result("certificate not valid yet");
    }

    cmp_result = X509_cmp_time(not_after, &now_utc);
    if (!cmp_result)
    {
        return operation::result("failed to read not_after: " + get_last_openssl_error());
    }

    if (cmp_result < 0)
    {
        return operation::result("certificate expired");
    }

    return operation::success;
}

operation::result certificate::write_pem(BIO* bio)
{
    assert(bio != nullptr);

    if (pkey_)
    {
        int write_ok =
            PEM_write_bio_PrivateKey(bio, pkey_.get(), nullptr, nullptr, 0, nullptr, nullptr);
        if (!write_ok)
        {
            return operation::result(
                "failed to write private key to bio: " + get_last_openssl_error());
        }
    }

    int write_ok = PEM_write_bio_X509(bio, cert_.get());
    if (!write_ok)
    {
        return operation::result("failed to write certificate to bio: " + get_last_openssl_error());
    }

    return operation::success;
}

operation::result certificate::read_x509_text(X509_NAME* name, int nid, string& text)
{
    int entry_index = X509_NAME_get_index_by_NID(name, nid, -1);
    if (entry_index < 0)
    {
        return operation::result("no entry in name for NID " + nid_description(nid));
    }

    X509_NAME_ENTRY* entry = X509_NAME_get_entry(name, entry_index);
    if (entry == nullptr)
    {
        return operation::result("entry for NID " + nid_description(nid) + " exists, but is null");
    }

    ASN1_STRING* asn_string = X509_NAME_ENTRY_get_data(entry);
    if (asn_string == nullptr)
    {
        return operation::result("data for NID " + nid_description(nid) + " exists, but is null");
    }

    // From openssl docs:
    // "In general it cannot be assumed that the data returned by
    // ASN1_STRING_data() is null terminated", therefore we use to_UTF8,
    // regardless of allocation overhead
    unsigned char* utf_text = nullptr;
    int utf_length = ASN1_STRING_to_UTF8(&utf_text, asn_string);
    if (utf_length < 0)
    {
        return operation::result(
            "data for NID " + nid_description(nid) + " is invalid: " + get_last_openssl_error());
    }

    text.assign(reinterpret_cast<const char*>(utf_text), utf_length);
    OPENSSL_free(utf_text);
    return operation::success;
}

bool contains_pem(const char* data, std::size_t size)
{
    static const std::string pem_prefix = "-----BEGIN CERTIFICATE-----\n";
    const char* data_end = data + size;
    return data_end != std::search(data, data_end, pem_prefix.begin(), pem_prefix.end());
}

operation::result parse_pem(const char* data, std::size_t size, X509*& cert)
{
    BIO* pem_bio;
    auto bio_result = create_readonly_bio(data, size, pem_bio);
    if (!bio_result)
    {
        return bio_result;
    }

    cert = PEM_read_bio_X509(pem_bio, nullptr, nullptr, nullptr);
    free_readonly_bio(pem_bio);

    if (cert == nullptr)
    {
        return operation::result(
            "failed to read certificate from PEM: " + get_last_openssl_error());
    }

    return operation::success;
}

operation::result parse_p12(
    const char* data,
    std::size_t size,
    const string& password,
    certificate_chain& certs)
{
    BIO* p12_bio;
    auto bio_result = create_readonly_bio(data, size, p12_bio);
    if (!bio_result)
    {
        return bio_result;
    }

    PKCS12* p12 = d2i_PKCS12_bio(p12_bio, nullptr);
    free_readonly_bio(p12_bio);

    if (!p12)
    {
        return operation::result("failed to parse P12 data: " + get_last_openssl_error());
    }

    X509* head_cert = nullptr;
    EVP_PKEY* private_key = nullptr;
    STACK_OF(X509)* tail_certs = nullptr;
    int parse_ok = PKCS12_parse(p12, password.c_str(), &private_key, &head_cert, &tail_certs);
    PKCS12_free(p12);

    if (!parse_ok)
    {
        if (head_cert) X509_free(head_cert);
        if (private_key) EVP_PKEY_free(private_key);
        if (tail_certs) sk_X509_pop_free(tail_certs, X509_free);
        return operation::result("failed to extract cert from p12: " + get_last_openssl_error());
    }

    certs.emplace_back(head_cert, private_key);
    while (sk_X509_num(tail_certs) > 0)
    {
        certs.emplace_back(sk_X509_pop(tail_certs));
    }
    sk_X509_free(tail_certs);

    return operation::success;
}

operation::result parse_private_key(const char* data, std::size_t size, EVP_PKEY*& private_key)
{
    BIO* private_key_bio;
    auto bio_result = create_readonly_bio(data, size, private_key_bio);
    if (!bio_result)
    {
        return bio_result;
    }

    private_key = PEM_read_bio_PrivateKey(private_key_bio, nullptr, nullptr, nullptr);
    free_readonly_bio(private_key_bio);

    if (private_key == nullptr)
    {
        return operation::result(
            "failed to read private key from PEM: " + get_last_openssl_error());
    }

    return operation::success;
}

operation::result write_pem(certificate_chain& certs, string& buffer)
{
    using unique_bio_ptr = std::unique_ptr<BIO, void (*)(BIO*)>;
    unique_bio_ptr bio{ BIO_new(BIO_s_mem()), BIO_vfree };
    if (!bio)
    {
        return operation::result("failed to create empty BIO: " + get_last_openssl_error());
    }

    for (auto& cert : certs)
    {
        auto write_result = cert.write_pem(bio.get());
        if (!write_result)
        {
            return operation::result(
                "failed to write certificate to PEM: " + get_last_openssl_error());
        }
    }

    char* bio_data;
    long bio_size = BIO_get_mem_data(bio.get(), &bio_data);
    buffer.assign(bio_data, static_cast<std::size_t>(bio_size));

    return operation::success;
}

}}
