import sys
import logging
from yql.api.v1.client import YqlClient

from locker import Locker
from mail.xiva.crm.src.util import YtPaths, cur_utc_ts, load_config
from mail.xiva.crm.src.user_events_db import UserEventsDBYT, FakeUserEventsDB
from mail.xiva.crm.src.miner import Miner
from mail.xiva.crm.src.executor import Executor
from mail.xiva.crm.src.scheduler import Scheduler, SchedulerDBYT, FakeSchedulerDB
from mail.xiva.crm.src.xiva import Xiva
from mail.xiva.crm.src.history import HistoryYT, FakeHistory
from mail.xiva.crm.src.local_storage import LocalStorage
from mail.xiva.crm.src.web import Web
from mail.xiva.crm.src.fake_locker import FakeLocker


def make_fake_schedule(users):
    return [{
        'id': user['device_id'],
        'execute_ts': cur_utc_ts(),
        'started': 0,
        'data': user
    } for user in users]


def init_logger(config):
    if 'log_path' in config:
        logging.basicConfig(
            level=logging.INFO,
            format="[%(asctime)s] [%(module)s,%(funcName)s] %(message)s",
            filename=config["log_path"],
            datefmt="%Y-%m-%d %H:%M:%S")
    else:
        logging.basicConfig(
            level=logging.INFO,
            format="[%(asctime)s] [%(module)s,%(funcName)s] %(message)s",
            datefmt="%Y-%m-%d %H:%M:%S")
    logging.getLogger("yql.client.request").setLevel(logging.ERROR)


def main():
    if len(sys.argv) < 2:
        print("usage {} <config>".format(sys.argv[0]))
        sys.exit(1)
    config = load_config(sys.argv[1])
    init_logger(config)
    client = YqlClient(db='hahn', token_path=config['yql_token_path'])
    yt_paths = YtPaths(config)
    if 'locker_config' in config:
        locker = Locker(config['locker_config'])
    else:
        locker = FakeLocker()
    if 'fakes' in config and 'scheduler_db' in config['fakes']:
        scheduler_db = FakeSchedulerDB()
    else:
        scheduler_db = SchedulerDBYT(client, yt_paths, locker)
    if 'fake_schedule' in config:
        scheduler_db.write_schedule(make_fake_schedule(config['fake_schedule']))
    if 'fakes' in config and 'history' in config['fakes']:
        history = FakeHistory()
    else:
        history = HistoryYT(client, yt_paths)
    if 'fakes' in config and 'user_events_db' in config['fakes']:
        user_events_db = FakeUserEventsDB()
    else:
        user_events_db = UserEventsDBYT(client, yt_paths)
    scheduler = Scheduler(scheduler_db, history, locker, config)
    xiva = Xiva(
        config['xiva']['url'],
        open(config['xiva']['token_path'], 'r').read().strip(),
        config['xiva']['wait_round_time'],
        locker)
    executor = Executor(config, scheduler, user_events_db, history, xiva)
    scheduler.set_exec_handler(lambda tasks: executor.execute_tasks(tasks))
    storage = LocalStorage(config['state_path'])
    miner = Miner(user_events_db, scheduler, storage, config)
    scheduler.set_mine_handler(lambda: miner.mine_if_needed())
    web = Web(config['control_port'], scheduler)
    web.start()
    scheduler.run()


if __name__ == "__main__":
    main()
