import copy
import json
import logging
import time
from mail.xiva.crm.src.util import calc_next_step_ts, cur_utc_ts


class Executor:
    def __init__(self, config, scheduler, user_events_db, history, xiva):
        self.scheduler = scheduler
        self.user_events_db = user_events_db
        self.history = history
        self.xiva = xiva
        self.tasks_to_schedule = []
        self.history_events = []
        self.config = config
        self.logger = logging.getLogger("global")

    def execute_tasks(self, tasks):
        irrelevant_devices = self.user_events_db.check_relevance([task['data'] for task in tasks])
        delayed_tasks = []
        for task in tasks:
            status = {}
            if task['data']['device_id'] in irrelevant_devices:
                status['step_status'] = 'skip'
                self.schedule_task(task, status)
                self.history_events.append({
                    'name': 'already_finished_task',
                    'data': json.dumps({'task_id': task['id']})})
            elif task['data']['locale'] not in self.current_notification(task)['body']:
                status['step_status'] = 'skip'
                self.schedule_task(task, status)
                self.history_events.append({
                    'name': 'no_text_for_locale',
                    'data': json.dumps({'task_id': task['id']})})
            elif 'begin_ts' in task['data'] and cur_utc_ts() < task['data']['begin_ts']:
                self.logger.info("add {} to delayed tasks".format(task['id']))
                delayed_tasks.append((task['data']['begin_ts'], task))
            elif 'end_ts' in task['data'] and cur_utc_ts() > task['data']['end_ts']:
                status['step_status'] = 'belated'
                self.schedule_task(task, status)
                self.history_events.append({
                    'name': 'belated_task',
                    'data': json.dumps({'task_id': task['id']})})
            else:
                status = self.send_notification(task)
                self.schedule_task(task, status)
        for (begin_ts, task) in sorted(delayed_tasks, key=lambda (ts, task): ts):
            if cur_utc_ts() < begin_ts:
                interval = int((begin_ts - cur_utc_ts()) / (10**6))
                self.logger.info("sleep for {}".format(interval))
                time.sleep(interval)
            status = self.send_notification(task)
            self.schedule_task(task, status)
        self.history.write(self.history_events)
        self.history_events = []
        self.scheduler.schedule_tasks(self.tasks_to_schedule)
        self.tasks_to_schedule = []

    def campaign_config(self, task):
        return self.config['campaigns'][task['data']['campaign']]

    def current_notification(self, task):
        return self.campaign_config(task)["notifications"][task['data']['step']]

    def send_notification(self, task):
        log_params = [
            "device_id={}".format(task['data']['device_id']),
            "uid={}".format(task['data']['uid']),
            "campaign={}".format(task['data']['campaign']),
            "step={}".format(task['data']['step'])
        ]
        title = self.current_notification(task)['title'][task['data']['locale']]
        body = self.current_notification(task)['body'][task['data']['locale']]
        deeplink = self.current_notification(task)['deeplink']
        collapse_id = "{}_{}_{}".format(task['data']['device_id'], task['data']['campaign'], task['data']['step'])
        log_params.append("collapse_id={}".format(collapse_id))
        ttl = None
        if 'ttl' in self.current_notification(task):
            ttl = self.current_notification(task)['ttl']
            log_params.append("ttl={}".format(ttl))
        self.logger.info("send to xiva start {}".format(" ".join(log_params)))
        resp = self.xiva.send(task['data']['uid'], task['data']['device_id'], title, body, deeplink, collapse_id, ttl)
        resp['task_id'] = task['id']
        resp['step'] = task['data']['step']
        self.history_events.append({
            'name': 'send_notification',
            'data': json.dumps(resp)})
        log_params.append("status_code={}".format(resp['response_status']))
        if 'codes' in resp:
            log_params.append("codes={}".format(resp['codes']))
        self.logger.info("send to xiva finish {}".format(" ".join(log_params)))
        return {'step_status': 'xiva_send', 'xiva_status': resp['response_status']}

    def schedule_task(self, task, status):
        retriable_error = status['step_status'] == 'xiva_send' and (
            status['xiva_status'] / 100 == 5 or status['xiva_status'] == 429)
        retry_limit_exceeded = 'attempt' in task['data'] and task['data']['attempt'] >= self.config['xiva']['retries']
        need_retry = status['step_status'] == 'belated' or (retriable_error and not retry_limit_exceeded)
        if need_retry:
            self.schedule_retry(task, increment_attempt=retriable_error)
        elif task['data']['step'] in self.campaign_config(task)['scenario']:
            self.schedule_next_step(task)

    def calc_next_task_ts(self, task, allow_cur_day):
        return calc_next_step_ts(
            self.campaign_config(task)['notification']['time'],
            self.config['step_check_time'],
            task['data']['timezone'],
            allow_cur_day=allow_cur_day)

    def schedule_next_step(self, task):
        data = copy.deepcopy(task['data'])
        data['attempt'] = 0
        data['step'] = self.campaign_config(task)['scenario'][task['data']['step']]
        allow_cur_day = self.campaign_config(task)['notification']['limits'] == 'no_limits'
        ts = self.calc_next_task_ts(task, allow_cur_day=allow_cur_day)
        data['begin_ts'] = ts['begin_ts']
        data['end_ts'] = ts['end_ts']
        self.tasks_to_schedule.append({
            'id': task['id'],
            'execute_ts': ts['execute_ts'],
            'data': data})

    def schedule_retry(self, task, increment_attempt):
        data = copy.deepcopy(task['data'])
        if increment_attempt:
            data['attempt'] = data['attempt'] + 1 if 'attempt' in data else 1
        else:
            data['attempt'] = data['attempt'] if 'attempt' in data else 0
        ts = self.calc_next_task_ts(task, allow_cur_day=True)
        data['begin_ts'] = ts['begin_ts']
        data['end_ts'] = ts['end_ts']
        self.tasks_to_schedule.append({
            'id': task['id'],
            'execute_ts': ts['execute_ts'],
            'data': data})


class FakeExecutor:
    def __init__(self):
        self.tasks = []

    def execute_tasks(self, tasks):
        self.tasks += tasks
