#include <ymod_lease/node.h>
#include <ymod_messenger/module.h>
#include <yplatform/application/config/loader.h>
#include <yplatform/util/weak_bind.h>
#include <boost/asio/io_service.hpp>
#include <boost/python.hpp>
#include <functional>
#include <memory>
#include <thread>
#include <mutex>

namespace ph = std::placeholders;

struct locker_impl:
    yplatform::log::contains_logger,
    std::enable_shared_from_this<locker_impl>
{
    void init(const std::string& config_path) {
        yplatform::ptree config;
        utils::config::loader::from_file(config_path, config);
        config = config.get_child("config");

        if (config.count("log")) {
            yplatform::log::init_file(io, config.get<std::string>("log"));
        } else {
            yplatform::log::init_console(io);
        }
        logger(yplatform::log::find(io, "global"));
        YLOG_L(info) << "locker initialization started";

        messenger = std::make_shared<ymod_messenger::module>(io, config.get_child("ymod_messenger"));
        yplatform::register_module(io, "ymod_messenger", messenger);

        resource_name = config.get<std::string>("resource");
        auto lease_config = config.get_child("ymod_lease");
        lease_config.put("netch_module", "ymod_messenger");
        lease = std::make_shared<ymod_lease::node>(io, lease_config);
        yplatform::register_module(io, "ymod_lease", lease);
        lease->bind(resource_name,
            yplatform::weak_bind(&locker_impl::on_busy, yplatform::shared_from(this), ph::_1, ph::_2, ph::_3, ph::_4),
            yplatform::weak_bind(&locker_impl::on_free, yplatform::shared_from(this), ph::_1));

        std::set<std::string> hosts;
        auto lease_cluster = config.equal_range("lease_cluster");
        for (auto& [name, node]: boost::make_iterator_range(lease_cluster)) {
            hosts.insert(node.data());
        }
        messenger->connect_to_cluster(hosts);
        lease->start_acquire_lease(resource_name);
        thread = std::make_unique<std::thread>([this]() mutable { io.run(); });
        YLOG_L(info) << "locker initialization finished";
    }

    void on_busy(const std::string& resource,
        const std::string& node_id,
        ymod_lease::ballot_t /*ballot*/,
        const std::string& /*value*/)
    {
        YLOG_L(info) << "busy resource: resource=" << resource << " owner=" << node_id;
        std::lock_guard<std::mutex> lock(mutex);
        if (resource == resource_name) {
            resource_owner = node_id;
        }
    }

    void on_free(const std::string& resource) {
        YLOG_L(info) << "free resource: " << resource;
        std::lock_guard<std::mutex> lock(mutex);
        if (resource == resource_name) {
            resource_owner = "";
        }
    }

    bool acquired() {
        std::lock_guard<std::mutex> lock(mutex);
        return resource_owner == lease->node_id();
    }

    std::string resource_name;
    std::string resource_owner;
    std::shared_ptr<ymod_messenger::module> messenger;
    std::shared_ptr<ylease::node> lease;
    boost::asio::io_service io;
    std::mutex mutex;
    std::unique_ptr<std::thread> thread;
};

struct locker {
    locker(const std::string& config_path) {
        impl = std::make_shared<locker_impl>();
        impl->init(config_path);
    }

    bool acquired() {
        return impl->acquired();
    }

    std::shared_ptr<locker_impl> impl;
};

BOOST_PYTHON_MODULE(locker)
{
    using namespace boost::python;
    class_<locker>("Locker", init<std::string>())
        .def("acquired", &locker::acquired);
}
