import datetime
import logging
from mail.xiva.crm.src.util import calc_next_step_ts


class Miner:
    def __init__(self, user_events_db, scheduler, storage, config):
        self.user_events_db = user_events_db
        self.scheduler = scheduler
        self.storage = storage
        self.config = config
        self.logger = logging.getLogger("global")

    def _make_new_device_task(self, user):
        if user['platform'] == 'android' and 'welcome_android' in self.config['campaigns']:
            campaign = 'welcome_android'
        elif user['platform'] and 'welcome_ios' in self.config['campaigns']:
            campaign = 'welcome_ios'
        else:
            raise Exception('unknown platform: {}'.format(user['platform']))
        ts = calc_next_step_ts(
            self.config['campaigns'][campaign]['notification']['time'],
            self.config['step_check_time'],
            user['timezone'],
            allow_cur_day=True)
        task = {
            'id': user['device_id'],
            'execute_ts': ts['execute_ts'],
            'data': {
                'device_id': user['device_id'],
                'uid': user['uid'],
                'locale': user['locale'],
                'timezone': user['timezone'],
                'step': self.config['campaigns'][campaign]['scenario']['initial'],
                'campaign': campaign,
                'platform': user['platform'],
                'begin_ts': ts['begin_ts'],
                'end_ts': ts['end_ts']
            }
        }
        return task

    def mine_if_needed(self):
        get_new_users_ts = self.storage.get_datetime('get_new_users_ts')
        if (get_new_users_ts is not None and get_new_users_ts.day == datetime.datetime.now().day) or datetime.datetime.now().hour <= 2:
            return
        self.logger.info("mining started")
        devices = self._mine_new_devices()
        self._schedule_tasks_for_new_devices(devices)
        self._mine_uids_for_existing_devices()
        self.storage.set_datetime('get_new_users_ts', datetime.datetime.now())
        self.logger.info("mining finished")

    def _mine_new_devices(self):
        devices = self.user_events_db.get_fresh_installs()
        self.user_events_db.get_uids_for_new_devices(devices)
        authorized_devices = [device for device in devices if device['uid'] is not None]
        self.user_events_db.write_new_devices(authorized_devices)
        return authorized_devices

    def _schedule_tasks_for_new_devices(self, devices):
        tasks = {'ios': [], 'android': []}
        for device in devices:
            try:
                task = self._make_new_device_task(device)
                if self.scheduler.have_task(task['id']):
                    self.logger.info('schedule already have task {}'.format(task['id']))
                else:
                    if device['platform'] in tasks:
                        tasks[device['platform']].append(task)
            except Exception as e:
                self.logger.error("make_new_user_task exception: " + str(e))
        tasks_to_schedule = \
            tasks['ios'][:int(len(tasks['ios']) * self.config['mine_new_devices_percent']['ios'] / 100.0)] + \
            tasks['android'][:int(len(tasks['android']) * self.config['mine_new_devices_percent']['android'] / 100.0)]
        for task in tasks_to_schedule:
            self.logger.info(
                "new_user device_id={} uid={} locale={} timezone={} platform={}".format(
                    task['data']['device_id'], task['data']['uid'], task['data']['locale'],
                    task['data']['timezone'], task['data']['platform']))
        self.scheduler.schedule_tasks(tasks_to_schedule)

    def _mine_uids_for_existing_devices(self):
        user_device = self.user_events_db.get_new_uids_for_existing_devices()
        self.user_events_db.write_uids_for_existing_devices(user_device)


class FakeMiner:
    def __init__(self):
        self.mine_called = False

    def mine_if_needed(self):
        self.mine_called = True
