import json
import logging
import time
import datetime
from mail.xiva.crm.src.util import cur_utc_ts, date_time_from_ts


class FakeSchedulerDB:
    def __init__(self):
        self.logger = logging.getLogger("global")
        self.schedule = []

    def read_schedule(self):
        return self.schedule

    def write_schedule(self, schedule):
        self.logger.info("write schedule {}".format(schedule))
        self.schedule = schedule


class SchedulerDBYT:
    def __init__(self, client, yt_paths, locker):
        self.client = client
        self.yt_paths = yt_paths
        self.locker = locker
        self.logger = logging.getLogger("global")

    def read_schedule(self):
        self.logger.info('read schedule...')
        table = self.client.read_table(self.yt_paths.SCHEDULE)
        schedule = []
        for row in table:
            task = {}
            for key, value in zip(table.column_names, row):
                if key == 'task_data':
                    task['data'] = json.loads(value)
                else:
                    task[key] = value
            schedule.append(task)
        return schedule

    def write_schedule(self, schedule):
        if not self.locker.acquired():
            raise Exception("write schedule error: lock not acquired")
        if len(schedule) == 0:
            return
        self.logger.info('write schedule...')
        data = [[task['id'], cur_utc_ts(), task['execute_ts'], task['started'], json.dumps(task['data'])] for task in schedule]
        column_names = ['id', 'ts', 'execute_ts', 'started', 'task_data']
        column_types = ['String', 'Timestamp', 'Timestamp', 'Int32', 'String']
        self.client.write_table(self.yt_paths.SCHEDULE, data, column_names, column_types)


class Scheduler:
    def __init__(self, scheduler_db, history, locker, config):
        self.scheduler_db = scheduler_db
        self.history = history
        self.stopped = False
        self.schedule = dict()
        self.locker = locker
        self.config = config
        self.logger = logging.getLogger("global")
        schedule_list = scheduler_db.read_schedule()
        for task in schedule_list:
            self.schedule[task['id']] = task

    def set_exec_handler(self, handler):
        self.on_exec = handler

    def set_mine_handler(self, handler):
        self.on_mine = handler

    def have_task(self, id):
        return id in self.schedule

    def start(self):
        self.stopped = False

    def stop(self):
        self.stopped = True

    def schedule_tasks(self, tasks):
        for task in tasks:
            if task['id'] in self.schedule:
                old_task = self.schedule[task['id']]
                old_task['execute_ts'] = task['execute_ts']
                old_task['started'] = 0
                old_task['ts'] = cur_utc_ts()
                old_task['data'] = task['data']
            else:
                task['started'] = 0
                task['ts'] = cur_utc_ts()
                self.schedule[task['id']] = task
        self.schedule = self.remove_expired_tasks(self.schedule)
        self.scheduler_db.write_schedule(self.schedule.values())

    def remove_expired_tasks(self, schedule):
        return {id: task for id, task in schedule.items() if not self.is_task_expired(task)}

    def is_task_expired(self, task):
        if not task['started']:
            return False
        execute_date = date_time_from_ts(task['execute_ts'])
        return (datetime.datetime.now() - execute_date).days > self.config['task_expire_interval_days']

    def run_once(self):
        if self.stopped:
            self.logger.info('skip iteration - stopped')
            return
        if not self.locker.acquired():
            self.logger.info('skip iteration - lock not acquired')
            return
        try:
            self.on_mine()
        except Exception as e:
            self.logger.error('mine exception: ' + str(e))
        self.logger.info('iteration started')
        try:
            tasks_to_start = []
            events = []
            for task in self.schedule.values():
                if task['started'] == 0 and task['execute_ts'] < cur_utc_ts():
                    task['started'] = 1
                    task['ts'] = cur_utc_ts()
                    tasks_to_start.append(task)
                    events.append({
                        'name': 'execute_task',
                        'data': json.dumps({'task_id': task['id']})
                    })
            if len(tasks_to_start) > 0:
                self.scheduler_db.write_schedule(self.schedule.values())
                self.history.write(events)
                self.on_exec(tasks_to_start)
        except Exception as e:
            self.logger.error('run exception: ' + str(e))
        self.logger.info('iteration finished')

    def run(self):
        while True:
            self.run_once()
            time.sleep(self.config['wait_interval'])


class FakeScheduler:
    def __init__(self):
        self.tasks = {}
        self.stopped = False

    def schedule_tasks(self, tasks):
        for task in tasks:
            self.tasks[task['id']] = task

    def have_task(self, id):
        return id in self.tasks
