import requests
import json
import datetime
import time
import logging


class Xiva:
    def __init__(self, xiva_url, xiva_token, wait_round_time, locker):
        self.xiva_url = xiva_url
        self.xiva_token = xiva_token
        self.wait_round_time = wait_round_time
        self.locker = locker
        self.logger = logging.getLogger("global")

    def send(self, uid, device_id, title, body, deeplink, collapse_id, ttl=None):
        if not self.locker.acquired():
            raise Exception('send to xiva error: lock not acquired')
        if self.wait_round_time:
            now = datetime.datetime.now()
            if now.minute % 5 == 0:
                interval = 61 - now.second
                self.logger.info("sleep for {}".format(interval))
                time.sleep(interval)
            elif now.minute % 5 == 4:
                interval = 121 - now.second
                self.logger.info("sleep for {}".format(interval))
                time.sleep(interval)
        params = [
            "user={}".format(uid),
            "event=xivacrm"
        ]
        if ttl is not None:
            params.append("ttl={}".format(ttl))
        url = "{}/v2/send?{}".format(self.xiva_url, "&".join(params))
        headers = {
            "Authorization": "Xiva {}".format(self.xiva_token),
            "Content-Type": "application/json",
            "X-DeliveryMode": "direct"
        }
        body = json.dumps({
            "payload": {
                "yamp": {
                    "b": False,
                    "d": {
                        "e": title,
                        "g": body,
                        "w": deeplink
                    },
                    "l": deeplink
                }
            },
            "repack": {
                "apns": {
                    "aps": {
                        "alert": {
                            "title": title,
                            "body": body
                        }
                    },
                    "collapse-id": collapse_id
                },
                "fcm": {
                    "repack_payload": ["yamp"]
                }
            },
            "subscriptions": [{"device": [device_id]}]
        })
        try:
            resp = requests.post(url=url, headers=headers, data=body)
            ret = {
                'response_status': resp.status_code,
                'response_text': resp.text,
                'request_text': body
            }
            resp_json = json.loads(resp.text)
            if type(resp_json) == list:
                for entry in resp_json:
                    if 'codes' in ret:
                        ret['codes'] += ',' + str(entry['code'])
                    else:
                        ret['codes'] = str(entry['code'])
            elif 'code' in resp_json:
                ret['codes'] = str(resp_json['code'])
            for header in ['Y-Context', 'TransitID', 'NotificationID']:
                if header in resp.headers:
                    ret[header.lower()] = resp.headers[header]
            return ret
        except Exception as e:
            return {'response_status': 0, 'response_text': str(e), 'codes': '0'}


class FakeXiva:
    def __init__(self):
        self.notifications = []
        self.response = {
            'response_status': 200,
            'response_text': '{"code":200}',
            'codes': '200'
        }

    def send(self, uid, device_id, title, body, deeplink, collapse_id, ttl=None):
        self.notifications.append({
            'uid': uid,
            'device_id': device_id,
            'title': title,
            'body': body,
            'deeplink': deeplink,
            'collapse_id': collapse_id
        })
        return self.response
