CREATE SCHEMA resharding;

CREATE TYPE resharding.migration AS (state bigint, end_gid bigint);

CREATE TABLE resharding.migrations
(
    gid_range int8range,
    state bigint NOT NULL,
    EXCLUDE USING gist (gid_range WITH &&)
);

INSERT INTO resharding.migrations (gid_range, state) VALUES('[0, 65535]'::int8range, 0);

CREATE OR REPLACE FUNCTION resharding.set_migration_state(i_gid bigint,
    i_allowed_states bigint[], i_state_to bigint)
RETURNS SETOF resharding.migration LANGUAGE plpgsql AS
$$
DECLARE
    range_from int8range;
    state_from bigint;
    num_found int;
    m record;
    prev resharding.migrations;
BEGIN
    LOCK TABLE resharding.migrations IN SHARE MODE;

    WITH deleted_range AS (
        DELETE FROM resharding.migrations
        WHERE gid_range @> i_gid AND state = ANY(i_allowed_states)
        RETURNING gid_range, state
    )
    SELECT gid_range, state FROM deleted_range
    INTO range_from, state_from;

    GET DIAGNOSTICS num_found = ROW_COUNT;
    IF num_found != 1 THEN
        -- Return a special tuple indicating that gid
        -- is in an inacceptable state
        RETURN QUERY VALUES (0::bigint, 65536::bigint);
        RETURN QUERY SELECT * FROM resharding.get_migrations_range();
        RETURN;
    END IF;

    INSERT INTO resharding.migrations VALUES
        (int8range(lower(range_from), i_gid, '[)'), state_from),
        (int8range(i_gid, i_gid, '[]'), i_state_to),
        (int8range(i_gid, upper(range_from), '()'), state_from);

    DELETE FROM resharding.migrations WHERE isempty(gid_range);

    prev := NULL;
    FOR m IN SELECT gid_range, state FROM resharding.migrations ORDER BY gid_range
    LOOP
        IF prev IS NOT NULL AND m.state = prev.state THEN
            DELETE FROM resharding.migrations
            WHERE gid_range = m.gid_range;

            UPDATE resharding.migrations
            SET gid_range = int8range(lower(gid_range), upper(m.gid_range), '[)')
            WHERE gid_range = prev.gid_range;

            prev.gid_range := int8range(lower(prev.gid_range), upper(m.gid_range), '[)');
        ELSE
            prev := m;
        END IF;

    END LOOP;

    RETURN QUERY SELECT * FROM resharding.get_migrations_range();
END;
$$;

CREATE OR REPLACE FUNCTION resharding.get_migration(i_gid bigint)
RETURNS bigint LANGUAGE SQL AS
$$
    SELECT state FROM resharding.migrations WHERE gid_range @> i_gid;
$$;

CREATE OR REPLACE FUNCTION resharding.get_migrations_range()
RETURNS SETOF resharding.migration LANGUAGE plpgsql AS
$$
BEGIN
    RETURN QUERY SELECT state, upper(gid_range) - 1 FROM resharding.migrations ORDER BY gid_range;
END;
$$;

CREATE OR REPLACE FUNCTION resharding.flip_states(state1 bigint, state2 bigint)
RETURNS VOID LANGUAGE SQL AS
$$
    WITH updated_ranges AS (
        UPDATE resharding.migrations
        SET state = state2
        WHERE state = state1
        RETURNING gid_range
    )
    UPDATE resharding.migrations
    SET state = state1
    WHERE state = state2
        AND gid_range NOT IN (SELECT * FROM updated_ranges);
$$;

GRANT USAGE ON SCHEMA resharding TO xiva_user;
GRANT ALL ON ALL TABLES IN SCHEMA resharding TO xiva_user;
GRANT EXECUTE ON ALL FUNCTIONS IN SCHEMA resharding TO xiva_user;
