DROP FUNCTION IF EXISTS xconf.put (xconf.conf_type, text, text, bytea, text);
DROP FUNCTION IF EXISTS xconf.put (xconf.conf_type, text, text, bytea, text, text);

CREATE OR REPLACE FUNCTION xconf.put (
    i_type xconf.conf_type,
    i_name text,
    i_owner_id text,
    i_settings bytea,
    i_token text,
    i_environment text = NULL,
    i_revision bigint = NULL
)
RETURNS xconf.put_list_type AS
$$
DECLARE
    res bigint;
    old_token text;
    old_owner_id text;
    old_revision bigint;
BEGIN
    SELECT token, owner_id, revision INTO old_token, old_owner_id, old_revision
    FROM xconf.configurations
    WHERE type = i_type AND name = i_name AND environment = i_environment
    FOR UPDATE;

    IF NOT FOUND THEN
        IF i_token IS NOT NULL THEN
            INSERT INTO xconf.tokens (token) VALUES (i_token);
        END IF;

        INSERT INTO xconf.configurations (type, name, settings, token, owner_id, environment)
        VALUES (i_type, i_name, i_settings, i_token, i_owner_id, i_environment)
        RETURNING revision INTO res;
    ELSE
        IF i_owner_id <> old_owner_id THEN
            RETURN (0, 'bad_owner_id')::xconf.put_list_type;
        END IF;
        IF i_revision IS NOT NULL AND i_revision < old_revision THEN
            RAISE NOTICE 'Revision outdated';
            RETURN (0::bigint, 'stale_revision'::xconf.error_code_type);
        END IF;
        IF i_token IS NOT NULL AND (old_token is NULL OR i_token <> old_token) THEN
            INSERT INTO xconf.tokens (token) VALUES (i_token);
        END IF;
        UPDATE xconf.configurations
        SET
            settings = i_settings,
            token = i_token,
            update_date = now(),
            revision = nextval('xconf.conf_rev_seq')
        WHERE
            type = i_type AND
            name = i_name AND
            environment = i_environment
        RETURNING revision INTO res;
    END IF;

    RETURN (res, NULL)::xconf.put_list_type;

EXCEPTION WHEN unique_violation THEN
    RETURN (0, 'token_violation')::xconf.put_list_type;
END;
$$ LANGUAGE PLPGSQL;

CREATE OR REPLACE FUNCTION xconf.update_owner_id (
    i_type xconf.conf_type,
    i_name text,
    i_new_owner_id text,
    i_environment text = NULL
)
RETURNS xconf.put_list_type AS
$$
DECLARE
    nfound bigint;
    res bigint;
BEGIN
    SELECT COUNT(*) INTO nfound
    FROM xconf.configurations
    WHERE type = i_type AND name = i_name AND environment = i_environment;

    IF nfound = 1 THEN
        UPDATE xconf.configurations
        SET
            owner_id = i_new_owner_id,
            update_date = now(),
            revision = nextval('xconf.conf_rev_seq')
        WHERE
            type = i_type AND
            name = i_name AND
            environment = i_environment
        RETURNING revision INTO res;

        RETURN (res, NULL)::xconf.put_list_type;
    ELSE
        RETURN (0, 'found_zero_or_several_apps')::xconf.put_list_type;
    END IF;
END;
$$ LANGUAGE PLPGSQL;
