#! /bin/bash

[[ "$TRACE" ]] && set -x
set -eo pipefail

function usage()
{
    echo 'usage: migrate_data [--restore-only] --from <conninfo> --dump <path> --to <conninfo> [--dump-jobs N] [--restore-jobs M]'
}

function echo_help()
{
    usage
    echo '
    This is xiva notifications migration script. It takes all table data
    from one pg database (shard) to another pg database (shard).
    It assumes, that:
    * source shard is not being written to
    * local host has enough disk space for the dump of all data in pg_dump directory format
    * destination shard has all necessary tables in place and empty (including partitions
    for xiva.notifications)'
}

while [[ -n "$1" ]]; do
    case $1 in
        -h | --help)
            echo_help
            exit
            ;;
        -f | --from)
            shift
            CONNINFO_FROM=$1
            ;;
        -d | --dump)
            shift
            DUMP_DIR=$1
            ;;
        -t | --to)
            shift
            CONNINFO_TO=$1
            ;;
        --dump-jobs)
            shift
            DUMP_JOBS=$1
            ;;
        --restore-jobs)
            shift
            RESTORE_JOBS=$1
            ;;
        --restore-only)
            RESTORE_ONLY="yes"
            ;;
        *)
            usage
            exit 1
            ;;
    esac
    shift
done

if [[ -z "$RESTORE_ONLY" ]]; then
    [[ -z "$CONNINFO_FROM" || -z "$DUMP_DIR" || -z "$CONNINFO_TO" ]] && usage && exit 1
else
    [[ -z "$DUMP_DIR" || -z "$CONNINFO_TO" ]] && usage && exit 1
fi

# Execute 1 job for each non-empty notifications partition by default.
[[ -z "$DUMP_JOBS" ]] && DUMP_JOBS=3
[[ -z "$RESTORE_JOBS" ]] && RESTORE_JOBS=3

if [[ -z "$RESTORE_ONLY" ]]; then
    DUMP_DIR="$DUMP_DIR/xstore_dump_`date '+%Y-%m-%d_%H-%M-%S'`"

    echo "dumping from $CONNINFO_FROM to $DUMP_FILE"

    pg_dump -v \
        --format=directory \
        --file="$DUMP_DIR" \
        --compress=9 \
        --jobs="$DUMP_JOBS" \
        --data-only \
        --schema=xiva \
        --exclude-table=xiva.subscriptions \
        --exclude-table=xiva.broken_subscriptions \
        --lock-wait-timeout=60 \
        "$CONNINFO_FROM"

    echo "dump finished at $DUMP_FILE"
fi

echo "restoring from $DUMP_DIR to $CONNINFO_TO"

# First restore services/counters sequentially, then notifications concurrently.
# We cannot restore notifications at the same time as services/counters, because
# notifications refer to counters.
LIST_FILE='/tmp/to_restore'
pg_restore -l "$DUMP_DIR" | fgrep \
    -e 'xiva services_sid_seq' \
    -e 'xiva services' \
    -e 'xiva counters' \
    > "$LIST_FILE"

pg_restore -v \
    --use-list="$LIST_FILE" \
    --format=directory \
    --data-only \
    --no-owner \
    --single-transaction \
    -d "$CONNINFO_TO" \
    "$DUMP_DIR"

pg_restore -l "$DUMP_DIR" | fgrep -v \
    -e 'xiva services_sid_seq' \
    -e 'xiva services' \
    -e 'xiva counters' \
    > "$LIST_FILE"

pg_restore -v \
    --use-list="$LIST_FILE" \
    --format=directory \
    --data-only \
    --jobs="$RESTORE_JOBS" \
    --no-owner \
    -d "$CONNINFO_TO" \
    "$DUMP_DIR"

echo "restore done on $CONNINFO_TO"
