CREATE FUNCTION code.add_notification(
    i_uid text,
    i_service text,
    i_event_dt timestamp with time zone,
    i_tag_names text[], -- unused
    i_extra_data text,
    i_ttl integer,
    i_content bytea,
    i_hash text,
    i_transit_id text,
    i_topic text DEFAULT NULL
) RETURNS xiva.add_notification_list_transit_type AS $$
DECLARE
    counters_inited boolean;
    is_valid boolean;
    new_local_id bigint;
    service_id bigint;
    start_of_day timestamp with time zone;
    end_of_day timestamp with time zone;
    earliest_listed_ts timestamp with time zone;
    updated_with_topic integer;
    merged_into_transit_id text;
BEGIN
    IF (i_event_dt IS NULL) THEN
        SELECT current_timestamp INTO i_event_dt;
    ELSE
        SELECT i_event_dt >= current_timestamp - '1 day'::interval
           AND i_event_dt <= current_timestamp + '1 day'::interval
            INTO is_valid;
        IF NOT is_valid THEN
            RAISE EXCEPTION 'Invalid event_dt value: %', i_event_dt;
        END IF;
    END IF;

    SELECT date_trunc('day', i_event_dt),
        date_trunc('day', i_event_dt + '1 day'::interval)
        INTO start_of_day, end_of_day;

    SELECT sid INTO service_id FROM xiva.services WHERE service_name = i_service;
    IF NOT FOUND THEN
        SELECT code.init_service(i_service) INTO service_id;
    END IF;

    SELECT COUNT(*)::int::boolean
        INTO counters_inited FROM xiva.counters
        WHERE uid = i_uid AND sid = service_id;

    IF NOT counters_inited THEN
        PERFORM code.init_counters(i_uid, service_id);
    END IF;

    SELECT next_local_id
        INTO new_local_id FROM xiva.counters
        WHERE uid = i_uid AND sid = service_id
        FOR UPDATE;

    updated_with_topic := 0;
    IF i_topic IS NOT NULL THEN
        -- Make advantage of constraint exclusion. 24 hour limit is taken
        -- from list_notifications_by_ids.
        SELECT current_timestamp - '24 hours'::interval INTO earliest_listed_ts;

        UPDATE xiva.notifications
            SET local_id = new_local_id, content = i_content, hash = i_hash,
                ttl = LEAST(ttl, i_ttl)
            WHERE sid = service_id AND uid = i_uid AND topic = i_topic
                AND event_dt >= earliest_listed_ts
            RETURNING transit_id INTO merged_into_transit_id;

        GET DIAGNOSTICS updated_with_topic = ROW_COUNT;
    END IF;

    IF updated_with_topic = 0 THEN
        BEGIN
            INSERT INTO xiva.notifications
                (uid, sid, local_id, event_dt,
                    extra_data, ttl, content, hash, transit_id, topic)
              VALUES (i_uid, service_id, new_local_id, i_event_dt,
                    i_extra_data, i_ttl, i_content, i_hash, i_transit_id, i_topic);
        EXCEPTION
            WHEN unique_violation THEN
                RAISE NOTICE 'Attempt to insert already stored notification.';
                SELECT n.local_id, n.transit_id, n.event_dt
                    INTO new_local_id, i_transit_id, i_event_dt
                    FROM xiva.notifications n
                    WHERE n.uid = i_uid
                        AND n.sid = service_id
                        AND n.hash = i_hash
                        AND n.event_dt >= start_of_day
                        AND n.event_dt < end_of_day;
                IF NOT FOUND THEN
                    RAISE EXCEPTION 'Something really strange happened.';
                END IF;
                RETURN (new_local_id, i_transit_id, ROUND(EXTRACT(EPOCH FROM i_event_dt))::bigint,
                    TRUE, 'duplicate')::xiva.add_notification_list_transit_type;
        END;
    END IF;

    UPDATE xiva.counters
        SET next_local_id = next_local_id + 1,
        total_count = total_count + 1,
        unseen_count = unseen_count + 1
      WHERE uid = i_uid AND sid = service_id;

    -- TODO: remove legacy is_duplicate flag from xiva.add_notification_list_transit_type
    IF updated_with_topic = 0 THEN
        RETURN (new_local_id, i_transit_id, ROUND(EXTRACT(EPOCH FROM i_event_dt))::bigint,
            FALSE, 'none')::xiva.add_notification_list_transit_type;
    ELSE
        RETURN (new_local_id, merged_into_transit_id, ROUND(EXTRACT(EPOCH FROM i_event_dt))::bigint,
            FALSE, 'topic')::xiva.add_notification_list_transit_type;
    END IF;

END;
$$ LANGUAGE plpgsql;
