CREATE OR REPLACE FUNCTION code.lift_messages(
    i_uid text,
    i_service text,
    i_local_id_from bigint,
    i_count bigint
) RETURNS bigint AS $$
DECLARE
    counters_inited boolean;
    base_local_id bigint;
    service_id bigint;
    earliest_listed_ts timestamp with time zone;
    updated_count integer;
BEGIN
    SELECT sid INTO service_id FROM xiva.services WHERE service_name = i_service;
    IF NOT FOUND THEN
        -- If service is not inited, there's no use trying to repeat message.
        RETURN 0;
    END IF;

    SELECT COUNT(*)::int::boolean INTO counters_inited FROM xiva.counters
        WHERE uid = i_uid AND sid = service_id;

    IF NOT counters_inited THEN
        -- If counters are not inited, there's no use trying to repeat message.
        RETURN 0;
    END IF;

    SELECT next_local_id
        INTO base_local_id FROM xiva.counters
        WHERE uid = i_uid AND sid = service_id
        FOR UPDATE;

    -- Make advantage of constraint exclusion. 24 hour limit is taken
    -- from list_notifications_by_ids.
    SELECT current_timestamp - '1 day'::interval INTO earliest_listed_ts;

    WITH repeated AS (
        SELECT local_id, ROW_NUMBER()
        OVER (ORDER BY local_id) AS num
        FROM xiva.notifications
        WHERE sid = service_id AND uid = i_uid AND event_dt >= earliest_listed_ts
            AND make_interval(secs => ttl - 5) > current_timestamp - event_dt
            AND local_id BETWEEN (i_local_id_from) AND (i_local_id_from + i_count - 1))
    UPDATE xiva.notifications
    -- ROW_NUMBER() starts from 1.
    SET local_id = base_local_id + repeated.num - 1
    FROM repeated
    WHERE notifications.local_id = repeated.local_id
        AND sid = service_id AND uid = i_uid AND event_dt >= earliest_listed_ts;

    GET DIAGNOSTICS updated_count = ROW_COUNT;

    IF updated_count > 0 THEN
        UPDATE xiva.counters
            SET next_local_id = next_local_id + updated_count,
            total_count = total_count + updated_count,
            unseen_count = unseen_count + updated_count
          WHERE uid = i_uid AND sid = service_id;
    END IF;

    RETURN updated_count;
END;
$$ LANGUAGE plpgsql;
