CREATE SCHEMA IF NOT EXISTS xiva;

CREATE TABLE xiva.services
(
    sid smallserial,
    service_name text not null,
    CONSTRAINT pk_services PRIMARY KEY (sid),
    CONSTRAINT uk_services_name UNIQUE (service_name)
);

CREATE TABLE xiva.counters
(
    uid bigint NOT NULL,
    sid smallint NOT NULL,
    total_count bigint NOT NULL DEFAULT 0,
    unseen_count bigint NOT NULL DEFAULT 0,
    next_local_id bigint NOT NULL DEFAULT 1,
    last_seen_id bigint NOT NULL DEFAULT 0,
    last_seen_dt timestamp with time zone,
    CONSTRAINT pk_counters PRIMARY KEY (uid, sid),
    CONSTRAINT fk_counters_sid_services FOREIGN KEY (sid)
        REFERENCES xiva.services ON DELETE RESTRICT,
    CONSTRAINT check_counters CHECK (
        total_count >= 0 AND
        unseen_count >= 0 AND
        total_count >= unseen_count AND
        total_count < next_local_id AND
        total_count >= last_seen_id
    )
);

CREATE TABLE xiva.tags
(
    tag_id serial,
    tag_name text not null,
    CONSTRAINT pk_tags PRIMARY KEY (tag_id),
    CONSTRAINT uk_tags UNIQUE (tag_name)
);

CREATE TABLE xiva.notifications
(
    uid bigint NOT NULL,
    sid smallint NOT NULL,
    local_id bigint NOT NULL,
    event_dt timestamp with time zone NOT NULL,
    tags integer[],
    is_deleted boolean not null default false,
    is_sticked boolean not null default false,
    delayed_until_dt timestamp with time zone,
    extra_data text,
    ttl integer NOT NULL,
    content bytea NOT NULL,
    hash text NOT NULL,
    CONSTRAINT pk_notifications PRIMARY KEY (uid, sid, local_id),
    CONSTRAINT uk_notifications_uid_sid_hash UNIQUE (uid, sid, hash),
    CONSTRAINT fk_notifications_uid_service_counters FOREIGN KEY (uid, sid)
        REFERENCES xiva.counters ON DELETE RESTRICT,
    CONSTRAINT check_delayed_date CHECK (
        (delayed_until_dt IS NULL) OR
        (
            delayed_until_dt > current_timestamp AND
            delayed_until_dt > event_dt AND
            delayed_until_dt < event_dt + (ttl || ' hours')::interval
        )
    )
);

CREATE TABLE xiva.notifications_sticked
(
    CONSTRAINT pk_notifications_sticked PRIMARY KEY (uid, sid, local_id),
    CONSTRAINT uk_notifications_sticked_uid_sid_hash UNIQUE (uid, sid, hash),
    CONSTRAINT fk_notifications_sticked_uid_service_counters FOREIGN KEY (uid, sid)
        REFERENCES xiva.counters ON DELETE RESTRICT,
    CONSTRAINT check_is_sticked_true CHECK (
        is_sticked IS TRUE
    )
)
INHERITS (xiva.notifications);

CREATE TYPE xiva.notifications_list_type AS (
    uid bigint, service text, local_id bigint, event_ts bigint,
    tags text[], ttl integer, content bytea, is_deleted boolean,
    is_sticked boolean, delayed_until_ts bigint
);

CREATE TYPE xiva.counters_list_type AS (
    uid bigint, service text, total_count bigint,
    unseen_count bigint, next_local_id bigint,
    last_seen_id bigint, last_seen_ts bigint
);

CREATE TYPE xiva.add_notification_list_type AS (
    local_id bigint, event_ts bigint, is_duplicate boolean
);
