DROP TABLE IF EXISTS xiva.notifications_sticked;
DROP TABLE IF EXISTS xiva.tags;


DROP TABLE IF EXISTS xiva.counters CASCADE;
DROP TABLE IF EXISTS xiva.notifications CASCADE;
DROP TABLE IF EXISTS xiva.services CASCADE;


CREATE TABLE xiva.services
(
    sid bigserial,
    service_name text not null,
    CONSTRAINT pk_services PRIMARY KEY (sid),
    CONSTRAINT uk_services_name UNIQUE (service_name)
);

CREATE TABLE xiva.counters
(
    uid text NOT NULL,
    sid bigint NOT NULL,
    total_count bigint NOT NULL DEFAULT 0,
    unseen_count bigint NOT NULL DEFAULT 0,
    next_local_id bigint NOT NULL DEFAULT 1,
    last_seen_id bigint NOT NULL DEFAULT 0,
    last_seen_dt timestamp with time zone,
    CONSTRAINT pk_counters PRIMARY KEY (uid, sid),
    CONSTRAINT fk_counters_sid_services FOREIGN KEY (sid)
        REFERENCES xiva.services ON DELETE RESTRICT,
    CONSTRAINT check_counters CHECK (
        total_count >= 0 AND
        unseen_count >= 0 AND
        total_count >= unseen_count AND
        total_count < next_local_id AND
        total_count >= last_seen_id
    ),
    CONSTRAINT check_text_len CHECK (char_length(uid) <= 256)
);

CREATE TABLE xiva.notifications
(
    uid text NOT NULL,
    sid bigint NOT NULL,
    local_id bigint NOT NULL,
    event_dt timestamp with time zone NOT NULL,
    is_deleted boolean not null default false,
    is_sticked boolean not null default false,
    delayed_until_dt timestamp with time zone,
    extra_data text,
    ttl integer NOT NULL,
    content bytea NOT NULL,
    hash text NOT NULL,
    CONSTRAINT check_delayed_date CHECK (
        (delayed_until_dt IS NULL) OR
        (
            delayed_until_dt > current_timestamp AND
            delayed_until_dt > event_dt AND
            delayed_until_dt < event_dt + (ttl || ' hours')::interval
        )
    ),
    CONSTRAINT check_text_len CHECK (char_length(uid) <= 256)
) PARTITION BY RANGE (event_dt);

-- Natively partitioned tables cannot have any keys
-- or reference any other tables in PG 10,
-- partman is able to create necessary constraints
-- on children through a "template table".
CREATE TABLE xiva.notifications_template
(
    LIKE xiva.notifications,
    CONSTRAINT pk_notifications PRIMARY KEY (uid, sid, local_id),
    CONSTRAINT uk_notifications_uid_sid_hash UNIQUE (uid, sid, hash),
    CONSTRAINT fk_notifications_uid_service_counters FOREIGN KEY (uid, sid)
        REFERENCES xiva.counters ON DELETE RESTRICT
);

DROP TYPE IF EXISTS xiva.notifications_list_type CASCADE;
DROP TYPE IF EXISTS xiva.counters_list_type CASCADE;

CREATE TYPE xiva.notifications_list_type AS (
    uid text, service text, local_id bigint, event_ts bigint,
    tags text[], ttl integer, content bytea, is_deleted boolean,
    is_sticked boolean, delayed_until_ts bigint
);

CREATE TYPE xiva.counters_list_type AS (
    uid text, service text, total_count bigint,
    unseen_count bigint, next_local_id bigint,
    last_seen_id bigint, last_seen_ts bigint
);

-- Partitioning; in schema public, because that is the way pgaas works.
SELECT create_parent('xiva.notifications', 'event_dt', 'native', 'daily', NULL, 15,
    p_jobmon := false, p_template_table := 'xiva.notifications_template');
UPDATE
    part_config
SET
    retention = '2 days',
    retention_keep_table = FALSE
WHERE
    parent_table = 'xiva.notifications';
