CREATE FUNCTION code.add_subscription(
    i_uid text, i_gid integer, i_service text, i_id text,
    i_callback text, i_filter text,
    i_extra_data text, i_client text,
    i_ttl integer, i_session_key text default null,
    i_init_local_id bigint default null,
    i_platform text default null,
    i_device text default null,
    i_bb_connection_id text default null,
    i_uidset text default null,
    i_synchronous_commit text default null
)
RETURNS integer LANGUAGE plpgsql AS
$$
BEGIN
    IF i_synchronous_commit IS NOT NULL THEN
        PERFORM set_config('synchronous_commit', i_synchronous_commit, true);
    END IF;

    INSERT INTO xiva.subscriptions(
            uid, gid, service, id, callback, filter,
            extra_data, client, ttl, session_key,
            init_local_id, ack_local_id, init_time,
            ack_time, platform, device, bb_connection_id,
            uidset)
        VALUES (
            i_uid, i_gid, i_service, i_id, i_callback, i_filter,
            i_extra_data, i_client, i_ttl, i_session_key,
            i_init_local_id, i_init_local_id,
            now(), now(), i_platform, i_device, i_bb_connection_id,
            i_uidset)
        ON CONFLICT (uid, service, id) DO UPDATE
        SET gid = i_gid,
            callback = i_callback,
            filter = i_filter,
            extra_data = i_extra_data,
            client = i_client,
            ttl = i_ttl,
            session_key = i_session_key,
            init_local_id = i_init_local_id,
            init_time = now(), ack_time = now(),
            ack_local_id = i_init_local_id,
            platform = i_platform,
            device = i_device,
            bb_connection_id = i_bb_connection_id,
            uidset = i_uidset,
            -- We cannot tell, whether we really need to reset retry state,
            -- but not resetting it when necessary (e.g. push token changed)
            -- is worse than resetting it when not necessary (e.g. filters changed)
            retry_interval = NULL,
            next_retry_time = NULL;

    RETURN 1;
END;
$$;

CREATE FUNCTION code.add_broken_subscriptions(
    i_platform text,
    i_id text [],
    i_report_time timestamp with time zone [],
    i_synchronous_commit text default null
)
RETURNS integer AS $$
BEGIN
    IF i_synchronous_commit IS NOT NULL THEN
        PERFORM set_config('synchronous_commit', i_synchronous_commit, true);
    END IF;

    BEGIN
        INSERT INTO xiva.broken_subscriptions(
            platform, id, report_time
        )
        VALUES (
            i_platform, UNNEST(i_id), UNNEST(i_report_time)
        );
        RETURN 1;
    EXCEPTION WHEN unique_violation THEN
        -- it's OK, ignore
    END;
    RETURN 0;
END;
$$ LANGUAGE plpgsql;

CREATE FUNCTION code.del_subscription(i_uid text, i_service text, i_id text,
    i_overlap int default 0, i_synchronous_commit text default null)
RETURNS integer LANGUAGE plpgsql AS
$$
DECLARE
    res integer;
BEGIN
    IF i_synchronous_commit IS NOT NULL THEN
        PERFORM set_config('synchronous_commit', i_synchronous_commit, true);
    END IF;

    DELETE FROM xiva.subscriptions
        WHERE uid=i_uid
            AND service=i_service
            AND id=i_id
            AND init_time + i_overlap * INTERVAL '1 second' <= now();
    GET DIAGNOSTICS res = ROW_COUNT;
    RETURN res;
END;
$$;

CREATE FUNCTION code.del_url(i_callback text, i_synchronous_commit text default null)
RETURNS integer LANGUAGE plpgsql AS
$$
DECLARE
    res integer;
BEGIN
    IF i_synchronous_commit IS NOT NULL THEN
        PERFORM set_config('synchronous_commit', i_synchronous_commit, true);
    END IF;

    DELETE FROM xiva.subscriptions
        WHERE callback=i_callback;
    GET DIAGNOSTICS res = ROW_COUNT;
    RETURN res;
END;
$$;

CREATE FUNCTION code.update_ack_retry (
    i_uid text,
    i_service text,
    i_id text,
    i_old_ack_local_id bigint,
    i_new_ack_local_id bigint,
    i_smart_notify boolean,
    i_retry_interval interval,
    i_ack_event_ts timestamptz,
    i_synchronous_commit text default null
)
RETURNS bigint LANGUAGE plpgsql AS
$$
DECLARE
    res int;
    c_ack_local_id bigint;
    c_smart_notify boolean;
    c_retry_interval interval;
    c_next_retry_time timestamptz;
BEGIN
    IF i_synchronous_commit IS NOT NULL THEN
        PERFORM set_config('synchronous_commit', i_synchronous_commit, true);
    END IF;

    SELECT ack_local_id, smart_notify, retry_interval
        INTO c_ack_local_id, c_smart_notify, c_retry_interval
        FROM xiva.subscriptions
        WHERE uid = i_uid
            AND service = i_service
            AND id = i_id;

    IF NOT FOUND THEN
        RETURN NULL;
    END IF;

    IF (c_ack_local_id != 0 AND c_ack_local_id != i_old_ack_local_id)
        OR (
            (c_ack_local_id = i_old_ack_local_id)
            AND (c_ack_local_id = i_new_ack_local_id)
            AND (c_smart_notify = i_smart_notify)
            AND ((c_retry_interval IS NULL AND i_retry_interval IS NULL)
                OR COALESCE(c_retry_interval = i_retry_interval, FALSE))
        ) THEN
        RETURN c_ack_local_id;
    END IF;

    SELECT
        CASE WHEN i_retry_interval IS NOT NULL THEN now() + i_retry_interval ELSE NULL END
    INTO c_next_retry_time;

    IF c_ack_local_id = i_new_ack_local_id THEN
        UPDATE xiva.subscriptions
        SET
            smart_notify = i_smart_notify,
            next_retry_time = c_next_retry_time,
            retry_interval = i_retry_interval
        WHERE uid = i_uid
            AND service = i_service
            AND id = i_id;
    ELSE
        UPDATE xiva.subscriptions
        SET
            ack_local_id = i_new_ack_local_id,
            ack_time = now(),
            smart_notify = i_smart_notify,
            next_retry_time = c_next_retry_time,
            retry_interval = i_retry_interval,
            ack_event_ts = i_ack_event_ts
        WHERE uid = i_uid
            AND service = i_service
            AND id = i_id;
    END IF;

    RETURN i_new_ack_local_id;
END;
$$;

CREATE FUNCTION code.bulk_update_ack(
    i_uid text[],
    i_service text[],
    i_id text[],
    i_old_ack_local_id bigint[],
    i_new_ack_local_id bigint[],
    i_smart_notify boolean[],
    i_synchronous_commit text default null
) RETURNS SETOF bigint AS $$
DECLARE
    n_uid text;
    n_service text;
    n_id text;
    n_old_ack_local_id bigint;
    n_new_ack_local_id bigint;
    n_smart_notify bigint;
    length int;
BEGIN
    length := array_length(i_uid, 1);
    FOR pos in 1..length LOOP
        RETURN QUERY SELECT *
            FROM code.update_ack_retry(i_uid[pos], i_service[pos],
                i_id[pos], i_old_ack_local_id[pos],
                i_new_ack_local_id[pos], i_smart_notify[pos], NULL, NULL,
                i_synchronous_commit);
    END LOOP;
    RETURN;
END;
$$ LANGUAGE plpgsql;

CREATE FUNCTION code.remove_old_subscriptions(
    i_first_uid text, i_last_uid text, i_synchronous_commit text default null
)
RETURNS int LANGUAGE plpgsql AS
$$
DECLARE
    res int;
BEGIN
    IF i_synchronous_commit IS NOT NULL THEN
        PERFORM set_config('synchronous_commit', i_synchronous_commit, true);
    END IF;

    DELETE FROM xiva.subscriptions
    WHERE uid BETWEEN i_first_uid AND i_last_uid
        AND now() - init_time >= make_interval(hours => ttl);

    GET DIAGNOSTICS res = ROW_COUNT;

    RETURN res;
END;
$$;

CREATE FUNCTION code.remove_broken_subscriptions(i_synchronous_commit text default null)
RETURNS int LANGUAGE plpgsql AS
$$
DECLARE
    result int;
BEGIN
    IF i_synchronous_commit IS NOT NULL THEN
        PERFORM set_config('synchronous_commit', i_synchronous_commit, true);
    END IF;

    WITH broken_deleted AS (
        DELETE FROM xiva.broken_subscriptions
        RETURNING *),
    subscriptions_deleted AS (
    DELETE FROM xiva.subscriptions s
    USING broken_deleted bd
    WHERE s.platform IS NOT NULL
        AND s.platform = bd.platform
        AND s.id = bd.id
        AND s.init_time <= bd.report_time
    RETURNING s.*)
    SELECT count(*) FROM subscriptions_deleted INTO result;
    RETURN result;
END;
$$;

CREATE FUNCTION code.return_uid_ranges()
RETURNS text[] LANGUAGE plpgsql AS
$$
DECLARE
    uid_ranges text[];
    first_uid text;
    last_uid text;
BEGIN
    SELECT min(uid), max(uid)
        INTO first_uid, last_uid
        FROM xiva.subscriptions;

    SELECT histogram_bounds
        INTO uid_ranges
        FROM pg_stats
        WHERE schemaname = 'xiva'
            AND tablename = 'subscriptions'
            AND attname = 'uid';

    PERFORM array_prepend(first_uid, uid_ranges);
    PERFORM array_append(uid_ranges, last_uid);

    RETURN uid_ranges;
END;
$$;

CREATE VIEW code.list_subscriptions AS
    SELECT
        uid,
        service,
        id,
        callback,
        filter,
        extra_data,
        client,
        ttl,
        session_key,
        init_local_id,
        extract(epoch from init_time)::int as init_time,
        ack_local_id,
        extract(epoch from ack_time)::int as ack_time,
        smart_notify,
        platform,
        device,
        bb_connection_id,
        uidset,
        extract(epoch from next_retry_time)::int as next_retry_time,
        extract(epoch from retry_interval)::int as retry_interval,
        extract(epoch from ack_event_ts)::int as ack_event_ts
    FROM
        xiva.subscriptions;

CREATE FUNCTION code.update_subscriptions_uidset(i_uidset text,
    i_service text,
    i_old_cb text,
    i_new_cb text,
    i_synchronous_commit text default null
    )
RETURNS int LANGUAGE plpgsql AS
$$
DECLARE
    res int;
BEGIN
    IF i_synchronous_commit IS NOT NULL THEN
        PERFORM set_config('synchronous_commit', i_synchronous_commit, true);
    END IF;

    UPDATE xiva.subscriptions
        SET callback = i_new_cb
        WHERE uidset = i_uidset
            AND service = i_service
            AND (i_old_cb IS NULL OR callback = i_old_cb);

    GET DIAGNOSTICS res = ROW_COUNT;
    RETURN res;
END;
$$;

CREATE FUNCTION code.update_subscription(i_uid text,
    i_service text,
    i_id text,
    i_old_cb text,
    i_new_cb text,
    i_synchronous_commit text default null)
RETURNS int LANGUAGE plpgsql AS
$$
DECLARE
    res int;
BEGIN
    IF i_synchronous_commit IS NOT NULL THEN
        PERFORM set_config('synchronous_commit', i_synchronous_commit, true);
    END IF;

    UPDATE xiva.subscriptions
        SET callback = i_new_cb
        WHERE uid = i_uid
            AND service = i_service
            AND id = i_id
            AND (i_old_cb IS NULL OR callback = i_old_cb);

    GET DIAGNOSTICS res = ROW_COUNT;
    RETURN res;
END;
$$;

CREATE FUNCTION code.batch_del_subscriptions(i_uid text,
    i_service text,
    i_ids text[],
    i_init_not_after timestamptz,
    i_synchronous_commit text default null)
RETURNS SETOF text LANGUAGE plpgsql AS
$$
DECLARE
    res int;
BEGIN
    IF i_synchronous_commit IS NOT NULL THEN
        PERFORM set_config('synchronous_commit', i_synchronous_commit, true);
    END IF;

    RETURN QUERY DELETE FROM xiva.subscriptions
    WHERE uid=i_uid
        AND service=i_service
        AND id=any(i_ids)
        AND date_trunc('seconds', init_time + interval '0.5 seconds') <= i_init_not_after
    RETURNING text(id);
    RETURN;
END;
$$;
