import random
import requests

def _find_migration_state(gid, migrations_json):
    migration = next(m for m in migrations_json
        if m['start_gid'] <= gid and gid <= m['end_gid'])
    return migration['state']

def _find_master_conninfo(gid, shards_json):
    shard = next(s for s in shards_json['shards']
        if s['start_gid'] <= gid and gid <= s['end_gid'])
    return shard['master']

class Cluster:
    def __init__(self, config):
        # Adding 1 to end of the range, because ranges are inclusive in hub config
        self.gid_range = range(config['gid_range'][0], config['gid_range'][1] + 1)
        self.hubs = [address for address in config['hubs']]

    def any_hub(self):
        return random.choice(self.hubs)

    def master(self):
        return next(hub for hub in self.hubs if self.__is_master(hub))

    def slaves(self):
        return [hub for hub in self.hubs if not self.__is_master(hub)]

    def resharding_status(self, gid):
        return self.__resharding_status(self.any_hub(), gid)

    def prepare_migration(self, gid):
        for slave in self.slaves():
            self.__prepare_migration(slave, gid, 'slave')
        self.__prepare_migration(self.master(), gid, 'master')

    def start_migration(self, gid, reaction):
        for slave in self.slaves():
            self.__start_migration(slave, gid, reaction, 'slave')
        self.__start_migration(self.master(), gid, reaction, 'master')

    def finalize_migration(self, gid):
        self.__finalize_migration(self.master(), gid, 'master')
        for slave in self.slaves():
            self.__finalize_migration(slave, gid, 'slave')

    def abort_migration(self, gid):
        self.__abort_migration(self.master(), gid, 'master')
        for slave in self.slaves():
            self.__abort_migration(slave, gid, 'slave')

    def try_abort_migration(self, gid):
        try:
            self.abort_migration(gid)
        except: pass

    def __resharding_status(self, hub, gid):
        # Include gid in url for logging purposes.
        response = self.__get(hub, '/resharding/xtable/status', gid_tag=gid)

        state = _find_migration_state(gid, response.json()['migrations'])
        shard_from = _find_master_conninfo(gid, response.json()['old_shards'])
        shard_to = _find_master_conninfo(gid, response.json()['new_shards'])

        return (state, shard_from, shard_to)

    def __abort_migration(self, hub, gid, role):
        self.__post(hub, '/resharding/xtable/abort_migration', gid=gid, role=role)

    def __prepare_migration(self, hub, gid, role):
        self.__post(hub, '/resharding/xtable/prepare_migration', gid=gid, role=role)

    def __start_migration(self, hub, gid, reaction, role):
        self.__post(hub, '/resharding/xtable/start_migration', gid=gid, request_reaction=reaction, role=role)

    def __finalize_migration(self, hub, gid, role):
        self.__post(hub, '/resharding/xtable/finalize_migration', gid=gid, role=role)

    def __is_master(self, hub):
        response = self.__get(hub, '/replica_status')
        return response.json()['control_leader']

    def __get(self, address, path, **params):
        return self.__call_http(requests.Session.get, address, path, **params)

    def __post(self, address, path, **params):
        return self.__call_http(requests.Session.post, address, path, **params)

    def __call_http(self, method, address, path, **params):
        session = requests.Session()
        adapter = requests.adapters.HTTPAdapter(max_retries=3)
        url = 'http://' + address + path
        session.mount(url, adapter)
        response = method(session, url, timeout=(connect_timeout, request_timeout), params=params)
        response.raise_for_status()
        return response
