import socket
import json
import logging

from urlparse import urlparse, parse_qs
from BaseHTTPServer import HTTPServer, BaseHTTPRequestHandler

class HTTPServerV6(HTTPServer):
    address_family = socket.AF_INET6
    allow_reuse_address = True

    def __init__(self, *args, **kwargs):
        HTTPServer.__init__(self, *args, **kwargs)
        self.paths = {}

    def bind(self, path, call, **argtypes):
        self.paths[path] = (call, argtypes)

    def serve_forever(self):
        logging.info('serving on %s', self.server_address)
        HTTPServer.serve_forever(self)

class RequestHandler(BaseHTTPRequestHandler):
    def log_message(self, format, *args):
        logging.info('%s | %s', self.client_address[0], format%args)

    def do_GET(self):
        self.__do('GET')

    def do_POST(self):
        self.__do('POST')

    def __do(self, http_method):
        parsed_uri = urlparse(self.path)

        path = parsed_uri.path

        paths = self.server.paths
        if path not in paths:
            self.send_error(404, 'method %s not found' % path)
            return

        call = paths[path][0]
        argtypes = paths[path][1]

        uri_params = parse_qs(parsed_uri.query)

        multivals = [name for name, values in uri_params.iteritems() if len(values) > 1]
        if (any(multivals)):
            self.send_error(400, 'parameters %s have multiple values' % multivals)
            return

        uri_params = {name: values[0] for name, values in uri_params.iteritems()}
        # Silently ignore unknown parameters.
        uri_params = {name: value for name, value in uri_params.iteritems() if name in argtypes}

        try:
            params = {name: argtypes[name](value) for name, value in uri_params.iteritems()}
        except ValueError as e:
            self.send_error(400, 'invalid parameter: %s' % str(e))
            return

        missing = [name for name in argtypes if name not in params]
        if (any(missing)):
            self.send_error(400, 'parameters %s are missing' % missing)
            return

        try:
            body = call(**params)
        except Exception as e:
            self.send_error(500, '%s: %s' % (type(e).__name__, str(e)))
        else:
            self.send_response(200)
            self.send_header('content-type', 'application/json')
            self.wfile.write('\r\n')
            json.dump(body, self.wfile, separators=(',\n', ': '))
            self.wfile.write('\r\n')

def server(config):
    return HTTPServerV6((config['host'], config['port']), RequestHandler)
