import psycopg2
import psycopg2.extras
from psycopg2.extras import NamedTupleCursor
from psycopg2 import sql
from collections import namedtuple

UidRange = namedtuple('UidRange', [
    'first',
    'last'
])

def _execute(conninfo, query, expect_result, *params):
    with psycopg2.connect(conninfo) as connection:
        connection.autocommit = True
        with connection.cursor(cursor_factory=NamedTupleCursor) as cursor:
            cursor.execute(query, params)
            if expect_result:
                return cursor.fetchall()

def execute_values(conninfo, query, values, page_size, expect_result=False):
    with psycopg2.connect(conninfo) as connection:
        connection.autocommit = True
        with connection.cursor(cursor_factory=NamedTupleCursor) as cursor:
            psycopg2.extras.execute_values(
                cursor, query, values, page_size=page_size)
            if expect_result:
                return cursor.fetchall()

def execute(conninfo, query, *params):
    return _execute(conninfo, query, True, *params)

def execute_no_result(conninfo, query, *params):
    return _execute(conninfo, query, False, *params)

def select_attribute_ranges(conninfo, schema, table, attribute):
    select_min_max_sql = sql.SQL('''
        SELECT min({attribute}), max({attribute})
        FROM {schema}.{table}
    ''')
    # According to docs, histogram_bounds is of type any_array,
    # which actually is not an array. Direct cast did not work,
    # so cast to string and back, performance overhead is not
    # significant, since this request is not executed often.
    select_bounds_sql = sql.SQL('''
        SELECT string_to_array(array_to_string(histogram_bounds, ','), ',')
        FROM pg_stats
        WHERE schemaname = {schema}
            AND tablename = {table}
            AND attname = {attribute};
    ''')
    with psycopg2.connect(conninfo) as connection, connection.cursor() as cursor:
            cursor.execute(select_min_max_sql.format(
                schema = sql.Identifier(schema),
                table = sql.Identifier(table),
                attribute = sql.Identifier(attribute)
            ))
            min_uid, max_uid = cursor.fetchall()[0]
            cursor.execute(select_bounds_sql.format(
                schema = sql.Literal(schema),
                table = sql.Literal(table),
                attribute = sql.Literal(attribute)
            ))
            # Augment histogram_bounds values with min and max
            # (since we want a set of ranges that covers all values,
            # and histogram_bounds do not guarantee this).
            bounds = cursor.fetchall()[0][0]
            if bounds[0] != min_uid:
                bounds = [min_uid] + bounds
            if bounds[-1] != max_uid:
                bounds = bounds + [max_uid]
            return [UidRange(bounds[i-1], bounds[i]) for i in range(1, len(bounds))]
