#! /usr/bin/env python3

import logging
import time
import argparse
import psycopg2

import shards
import xtable
import xstore
import sys

def setup_logging(op_name, log_stdout):
    LOGFORMAT = '%(asctime)s %(message)s'
    if log_stdout:
        loghandler = logging.StreamHandler(sys.stdout)
    else:
        loghandler = logging.FileHandler(
            '/var/log/xivadba/{}.log'.format(op_name))
    logging.basicConfig(format=LOGFORMAT,
                        level=logging.INFO,
                        handlers=(loghandler,))
    logging.getLogger("requests").setLevel(logging.WARNING)
    logging.getLogger("urllib3").setLevel(logging.WARNING)

def run_partman_maintenance(log_stdout):
    setup_logging('run_partman_maintenance', log_stdout)
    try:
        shards_list = shards.get_unique('xstore')
    except Exception as e:
        logging.error('failed to get shards list: %s %s', type(e).__name__, str(e))
        raise

    for shard in shards_list:
        try:
            admin_conninfo = shards.admin_conninfo(shard)
            with psycopg2.connect(admin_conninfo) as connection:
                connection.autocommit = True
                with connection.cursor() as cursor:
                    cursor.execute('SET lock_timeout to 10000')
                    cursor.execute('''SELECT public.run_maintenance(
                        p_analyze := FALSE, p_jobmon := FALSE)''')
                    cursor.execute('ANALYZE xiva.notifications')
            logging.info('%s run_maintenance() OK', shards.friendly_name(shard))
        except Exception as e:
            logging.error('%s failed to run_maintenance: %s %s',
                shards.friendly_name(shard), type(e).__name__, str(e))

def masters_changed(old_shards, new_shards):
    old_masters = {s['master'] for s in old_shards}
    new_masters = {s['master'] for s in new_shards}
    return old_masters != new_masters

def janitor(db_type, op_name, log_stdout):
    db_module = globals()[db_type]
    op = getattr(db_module, op_name)
    setup_logging(op_name, log_stdout)
    logging.info('start')
    pool = None
    try:
        shards_list = shards.get_unique(db_type)

        pool = shards.Pool(shards_list, op, 5.0)
        pool.start()

        shards_changed = False
        while not shards_changed:
            time.sleep(600)
            new_shards_list = shards.get(db_type)
            shards_changed = masters_changed(shards_list, new_shards_list)
    except Exception as e:
        logging.exception('failed to run janitor: %s %s', type(e).__name__, str(e))
    finally:
        logging.info('stopping')
        if pool is not None:
            pool.stop()
        logging.info('stopped')

def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--log_stdout', dest="log_stdout", action='store_true')
    parser.add_argument('--remove_bad_fcm_subscriptions', dest="remove_bad_fcm_subscriptions", action='store_true')
    parser.add_argument('--remove_old_subscriptions', dest="remove_old", action='store_true')
    parser.add_argument('--remove_broken_subscriptions', dest="remove_broken", action='store_true')
    parser.add_argument('--remove_unused_counters', dest="remove_unused_counters", action='store_true')
    parser.add_argument('--sync_acks_with_counters', dest="sync_acks_with_counters", action='store_true')
    parser.add_argument('--run_partman_maintenance', dest="run_partman_maintenance", action='store_true')
    args = parser.parse_args()

    if args.remove_bad_fcm_subscriptions:
        janitor('xtable', 'remove_bad_fcm_subscriptions', args.log_stdout)
    elif args.remove_old:
        janitor('xtable', 'remove_old_subscriptions', args.log_stdout)
    elif args.remove_broken:
        janitor('xtable', 'remove_broken_subscriptions', args.log_stdout)
    elif args.remove_unused_counters:
        janitor('xstore', 'remove_unused_counters', args.log_stdout)
    elif args.sync_acks_with_counters:
        janitor('xtable', 'sync_acks_with_counters', args.log_stdout)
    elif args.run_partman_maintenance:
        run_partman_maintenance(args.log_stdout)
    else:
        raise Exception('No operation specified!')


if __name__ == '__main__':
    main()

