#pragma once

#include <map>
#include <sstream>
#include <boost/shared_ptr.hpp>
#include <boost/asio/error.hpp>
#include <boost/enable_shared_from_this.hpp>

#include <yplatform/log.h>

#include <equalizer/processor.h>
#include <processor/equalizer.h>

namespace yxiva { namespace equalizer {

class equalizer_proxy
    : public boost::enable_shared_from_this<equalizer_proxy>
    , public yplatform::log::contains_logger
{
    typedef notifications_sender sender_t;
    typedef std::shared_ptr<sender_t> sender_ptr_t;

    context_ptr ctx_;
    sender_ptr_t sender_;
    size_t last_stream_id = 0;
    mutex mutex_;

public:
    equalizer_proxy(
        boost::asio::io_service& io,
        const string& db_name,
        const equalizer_settings& st,
        const std::shared_ptr<yhttp::client>& http_client,
        const yplatform::log::source& logger = yplatform::log::source())
        : yplatform::log::contains_logger(logger)
        , ctx_(new context())
        , sender_(
              std::make_shared<notifications_sender>(io, st.sender, db_name, http_client, logger))
    {
    }

    void on_operation(const operation& op, equalizer_cb_t callback)
    {
        operation_ptr part_ptr = std::make_shared<operation>(op);
        on_operation(part_ptr, callback);
    }

    void on_operation(operation_ptr op, equalizer_cb_t callback)
    {
        scoped_lock lock(mutex_);

        if (last_stream_id >= op->stream_id)
        {
            callback(boost::system::error_code(), operation_state::seen);
        }
        else
        {
            last_stream_id = op->stream_id;
            sender_->send_notification(op, [cb = std::move(callback)]() {
                cb(boost::system::error_code(), operation_state::finished);
            });
        }
    }

    void cancel()
    {
        scoped_lock lock(mutex_);
        sender_->stop();
    }
};

}}
