#include "impl.h"

#include <algorithm>
#include <boost/asio.hpp>

#include <yplatform/find.h>
#include <ymod_webserver/codes.h>

#include <equalizer/operation.h>
#include <processor/equalizer.h>
#include <processor/simple_printer.h>
#include <processor/uid_fetcher.h>
#include <processor/fill_defaults.h>
#include <postprocessor/position/create_syncer.h>

namespace yxiva { namespace equalizer {

void impl::init(const yplatform::ptree& xml)
{
    paused_ = !xml.get("autostart", false);
    auto http_reactor = yplatform::find<yplatform::reactor>("httpclient");

    settings_ = load_equalizer_settings(xml);
    stats_ = boost::make_shared<module_stats>();

    if (settings_.use_selector)
    {
        lease_node_ = yplatform::find<ylease::node>("lease-node");
        http_client_ = std::make_shared<yhttp::client>(*http_reactor, settings_.http);

        select_position_syncer_ = create_syncer(settings_.syncer, lease_node_);
        position_holder_ = std::make_shared<position_holder_t>(select_position_syncer_);
        conninfo_provider_ = std::make_shared<pg_conninfo_provider>(
            settings_.pg_conn, global_io(), [this] { this->reload_db_list(); });
        selector_factory::init(position_holder_, conninfo_provider_);
        fix_active_db_list(settings_.db_list, db_aliases_, settings_.ignore_db_list);
        stats_->update_db_alias_dict(db_aliases_);
    }
}

void impl::reload(const yplatform::ptree& xml)
{
    auto updated_settings = load_equalizer_settings(xml);
    bool mode_changed = updated_settings.use_selector != settings_.use_selector;
    std::map<string, string> aliases;

    if (mode_changed)
    {
        YLOG_L(error) << "It's a bad idea to change working mode on the run";
        return;
    }

    if (not updated_settings.use_selector)
    {
        YLOG_L(warning) << "Reload in this mode is not supported";
        return;
    }

    fix_active_db_list(updated_settings.db_list, aliases, updated_settings.ignore_db_list);
    if (updated_settings.db_list != settings_.db_list)
    {
        update_active_locks(updated_settings.db_list);
    }

    conninfo_provider_->reload(updated_settings.pg_conn);

    {
        scoped_lock guard(mutex_);
        settings_ = updated_settings;
        db_aliases_ = aliases;
        stats_->update_db_alias_dict(aliases);
    }

    YLOG_L(info) << "reloaded";
}

void impl::reload_db_list()
{
    scoped_lock guard(mutex_);
    auto updated_db_list = settings_.db_list;
    auto ignored = settings_.ignore_db_list;
    auto aliases = db_aliases_;
    guard.unlock();

    fix_active_db_list(updated_db_list, aliases, ignored);
    update_active_locks(updated_db_list);

    guard.lock();
    settings_.db_list = updated_db_list;
    db_aliases_ = aliases;
    stats_->update_db_alias_dict(aliases);
}

void impl::fix_active_db_list(
    std::set<string>& base_db_list,
    std::map<string, string>& aliases,
    const std::set<string>& ignored_db_list) const
{
    bool ignore_new_dbs = ignored_db_list.count("*");

    for (auto db_name : base_db_list)
    {
        if (db_name.substr(0, 3) == "mdb") aliases[db_name] = db_name;
    }

    auto pg_dbs = conninfo_provider_->all_databases();
    for (auto& name : pg_dbs)
    {
        if (!ignore_new_dbs && !ignored_db_list.count(name.id)) base_db_list.insert(name.id);
        aliases[name.id] = name.alias;
    }
}

void impl::update_active_locks(std::set<string>& new_db_list)
{
    std::vector<string> missing_dbs, added_dbs;
    std::set_difference(
        settings_.db_list.begin(),
        settings_.db_list.end(),
        new_db_list.begin(),
        new_db_list.end(),
        std::inserter(missing_dbs, missing_dbs.end()));

    std::set_difference(
        new_db_list.begin(),
        new_db_list.end(),
        settings_.db_list.begin(),
        settings_.db_list.end(),
        std::inserter(added_dbs, added_dbs.end()));

    for (auto& db : missing_dbs)
        L_(info) << "update_active_locks release db: " << db;
    lock_manager_->unlock(missing_dbs);
    for (auto& db : added_dbs)
        L_(info) << "update_active_locks new db: " << db;
    lock_manager_->lock(added_dbs);
}

void impl::fini()
{
    if (settings_.use_selector)
    {
        selector_factory::fini();
    }
}

void impl::start()
{
    stopped_ = false;
    if (settings_.use_selector)
    {
        conninfo_provider_->start();
        auto on_locked_cb = boost::bind(&impl::on_locked, this, _1);
        auto on_unlocked_cb = boost::bind(&impl::on_unlocked, this, _1);
        auto on_updated_cb = boost::bind(&position_holder_t::update, position_holder_, _1, _2);
        lock_manager_ = yplatform::find<lock_manager>("lock-manager");
        lock_manager_->bind(on_locked_cb, on_unlocked_cb, on_updated_cb);
        lock_manager_->lock(
            std::vector<std::string>(settings_.db_list.begin(), settings_.db_list.end()));
    }
}

void impl::stop()
{
    scoped_lock guard(mutex_);
    stopped_ = true;
    paused_ = true;
    select_position_syncer_.reset();
    equalizers_.clear();
    if (conninfo_provider_)
    {
        conninfo_provider_->stop();
        conninfo_provider_.reset();
    }

    if (!plans_.empty())
    {
        std::map<string, plan_t> old_plans;
        std::swap(old_plans, plans_);
        guard.unlock();

        for (auto& plan : old_plans)
        {
            plan.second.stop();
        }
    }
}

void impl::manualy_put_operation(
    const string& sequence_id,
    const operation& op,
    equalizer_cb_t callback)
{
    // @todo: remove equalizer instances by timeout when sequence wasn't updated for too long
    // [when consumers have redistributed sequences if it's possible]
    equalizer_proxy_ptr equalizer = find_equalizer_proxy(sequence_id);
    try
    {
        equalizer->on_operation(op, callback);
    }
    catch (const std::exception& ex)
    {
        YLOG_CTX_LOCAL(op.ctx, error) << "error: " << ex.what();
        equalizer->cancel();
    }
    catch (...)
    {
        YLOG_CTX_LOCAL(op.ctx, error) << "unknown error";
        equalizer->cancel();
    }
}

json_value impl::full_state()
{
    scoped_lock guard(mutex_);

    json_value state(json_type::tobject);
    for (auto& plan : plans_)
    {
        json_value pipeline_state;
        auto&& processors = pipeline_state["processors"];
        processors.set_array();
        pipeline_state["started"] = !plan.second.is_stopped();
        pipeline_state["paused"] = plan.second.generator()->is_stopped();
        for (auto& proc : plan.second.processors())
        {
            json_value proc_state;
            proc_state["name"] = proc->name();
            proc_state["stream.size"] = proc->input()->buffer_size();
            proc_state["stream.commited_size"] = proc->input()->total_commited_size();
            proc_state["stream.id_offset"] = proc->input()->begin_id();
            processors.push_back(proc_state);
        }
        pipeline_state["name"] = db_aliases_[plan.first];
        state[plan.first] = pipeline_state;
    }
    return state;
}

equalizer_proxy_ptr impl::find_equalizer_proxy(const string& sequence_id)
{
    scoped_lock guard(mutex_);
    auto equalizer_it = equalizers_.find(sequence_id);
    if (equalizer_it == equalizers_.end())
    {
        auto equalizer = boost::make_shared<equalizer_t>(
            local_io(), sequence_id, settings_.eq, http_client_, logger());
        return equalizers_.insert(equalizer_it, std::make_pair(sequence_id, equalizer))->second;
    }
    return equalizer_it->second;
}

void impl::on_locked(const string& db_name)
{
    scoped_lock guard(mutex_);
    if (stopped_) return;
    if (plans_.count(db_name) == 0)
    {
        try
        {
            auto iplan = create_plan(db_name);
            if (!paused_) iplan->second.start();
            stats_->init_stats(db_name);
        }
        catch (const std::exception& ex)
        {
            YLOG_L(error) << "error executing plan: db_name=\"" << db_name
                          << "\""
                             " exception=\""
                          << ex.what() << "\"";
        }
        catch (...)
        {
            YLOG_L(error) << "error executing plan: db_name=\"" << db_name
                          << "\""
                             " exception=\"unknown\"";
        }
    }
}

void impl::on_unlocked(const string& db_name)
{
    scoped_lock guard(mutex_);
    if (stopped_) return;
    auto iplan = plans_.find(db_name);
    if (iplan != plans_.end())
    {
        auto plan = iplan->second;
        plans_.erase(iplan);
        stats_->drop_stats(db_name);
        guard.unlock();

        YLOG_L(info) << "stop executing plan for: db_name=\"" << db_name << "\"";
        plan.stop();
        plan.clear();
    }
}

impl::plan_iterator impl::create_plan(const string& db_name)
{
    YLOG_L(info) << "creating pg-plan for: db_name=\"" << db_name << "\"";
    return plans_.insert(plans_.end(), std::make_pair(db_name, create_pg_plan(db_name)));
}

impl::plan_t impl::create_pg_plan(const string& db_name) const
{
    auto generator =
        selector_factory::create_pg_selector(local_io(), settings_.selector, db_name, logger());
    auto fill_defaults_proc =
        std::make_shared<fill_defaults>(local_io(), settings_.stream, db_name, logger());
    auto uid_fetcher_proc =
        std::make_shared<uid_fetcher>(local_io(), settings_.auth, create_authorizer(), logger());
    auto equalizer_proc =
        std::make_shared<equalizer>(local_io(), settings_.eq, db_name, http_client_, logger());
    auto position_updater =
        std::make_shared<select_position_updater>(position_holder_, db_name, stats_);

    auto plan = pipeline::from<operation_parts_stream_t>(generator) | fill_defaults_proc |
        uid_fetcher_proc | equalizer_proc | position_updater;

    return plan;
}

authorizer_ptr impl::create_authorizer() const
{
    return boost::make_shared<caching_authorizer>(
        boost::make_shared<authorizer_blackbox>(), settings_.auth.cache_size);
}

void impl::start_generator()
{
    scoped_lock guard(mutex_);
    if (stopped_) return;
    if (!paused_)
    {
        YLOG_L(warning) << "start failed: service already started";
        return;
    }
    paused_ = false;
    for (auto& plan : plans_)
    {
        // Plan can be stopped if paused_ was true when that plan was created.
        // If paused_ was set to true after plan was created then only it's generator would be
        // paused.
        if (plan.second.is_stopped())
        {
            plan.second.start();
        }
        else
        {
            plan.second.generator()->resume();
        }
    }
    YLOG_L(info) << "equalizer started";
}

void impl::stop_generator()
{
    scoped_lock guard(mutex_);
    if (stopped_) return;
    if (paused_)
    {
        YLOG_L(warning) << "stop failed: service not started";
        return;
    }
    paused_ = true;
    for (auto& plan : plans_)
    {
        // plan is supposed to be started later, so only generator should be paused for
        //   not loosing data in streams because of reseting callbacks
        plan.second.generator()->pause();
    }
    YLOG_L(info) << "equalizer stopped";
}

void impl::drop_shard(const string& id, const string& alias)
{
    scoped_lock guard(mutex_);
    if (stopped_) return;
    std::vector<string> names;
    if (id.size())
    {
        names.push_back(id);
    }
    else if (alias.size())
    {
        for (auto& db : db_aliases_)
        {
            if (db.second == alias)
            {
                names.push_back(db.first);
            }
        }
    }
    guard.unlock();
    if (names.empty())
    {
        lock_manager_->read_only(
            std::vector<std::string>(settings_.db_list.begin(), settings_.db_list.end()),
            settings_.read_only_interval);
    }
    else
    {
        lock_manager_->read_only(names, settings_.read_only_interval);
    }
}

}}

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(yxiva::equalizer::impl)
