#pragma once

#include <boost/shared_ptr.hpp>

#include <yplatform/module.h>
#include <yplatform/reactor.h>
#include <ymod_lease/ymod_lease.h>
#include <ymod_httpclient/client.h>
#include <yxiva/core/authorizer.h>

#include <postprocessor/position/position_holder.h>
#include <postprocessor/select_position_updater.h>
#include <processor/pipeline_types.h>
#include <generator/selector_factory.h>
#include <generator/pg_conninfo_provider.h>
#include <lock_manager/lock_manager.h>

#include "processor.h"
#include "equalizer_proxy.h"
#include "settings.h"
#include "module_stats.h"

namespace yxiva { namespace equalizer {

typedef equalizer_proxy equalizer_t;
typedef boost::shared_ptr<equalizer_t> equalizer_proxy_ptr;
typedef std::map<string, equalizer_proxy_ptr> equalizers_map_t;

class impl
    : public yplatform::module
    , public processor
{
    typedef selector_factory::position_holder_t position_holder_t;

    typedef operation_parts_plan_t plan_t;
    typedef std::map<string, plan_t>::iterator plan_iterator;

public:
    void init(const yplatform::ptree& xml);
    void reload(const yplatform::ptree& xml);
    void fini();

    void start();
    void stop();

    boost::asio::io_service& global_io() const
    {
        return *yplatform::global_net_reactor->io();
    }

    boost::asio::io_service& local_io() const
    {
        try
        {
            auto reactor = yplatform::global_reactor_set->get("stream");
            return *reactor->io();
        }
        catch (const std::exception& ex)
        {
            YLOG_L(error) << "exception=\"" << ex.what() << "\" caught at " << __FILE__ << ":"
                          << __LINE__;
        }
        return global_io();
    }

    const yplatform::module_stats_ptr get_module_stats() const override
    {
        return stats_;
    }

    void start_generator() override;
    void stop_generator() override;
    void drop_shard(const string& name, const string& alias) override;

    void manualy_put_operation(
        const string& sequence_id,
        const operation& op,
        equalizer_cb_t callback) override;
    json_value full_state() override;

private:
    equalizer_proxy_ptr find_equalizer_proxy(const string& queue_id);

    void reload_db_list();
    void fix_active_db_list(
        std::set<string>& db_list,
        std::map<string, string>& aliases,
        const std::set<string>& ignored_db_list) const;
    void update_active_locks(std::set<string>& new_db_list);

    void on_locked(const string& db_name);
    void on_unlocked(const string& db_name);
    plan_iterator create_plan(const string& db_name);
    plan_t create_pg_plan(const string& db_name) const;

    authorizer_ptr create_authorizer() const;

private:
    equalizers_map_t equalizers_;
    std::shared_ptr<yhttp::client> http_client_;
    std::map<string, plan_t> plans_;
    module_stats_ptr stats_;
    mutex mutex_;
    settings settings_;
    std::map<string, string> db_aliases_;

    std::shared_ptr<pg_conninfo_provider> conninfo_provider_;
    std::shared_ptr<position_holder_t> position_holder_;
    position_holder_t::sync_strategy_ptr select_position_syncer_;
    boost::shared_ptr<ylease::node> lease_node_;
    boost::shared_ptr<lock_manager> lock_manager_;
    string my_host_ip_;
    bool paused_;
    bool stopped_;
};

}}
