#pragma once

#include <atomic>
#include <deque>
#include <unordered_set>
#include <unordered_map>
#include <boost/asio/strand.hpp>

#include <yplatform/find.h>
#include <yplatform/util/safe_call.h>
#include <ymod_httpclient/call.h>
#include <yxiva/core/http_handlers.h>
#include <yxiva/core/json.h>
#include <yxiva/core/services/names.h>

#include <equalizer/context.h>
#include <equalizer/operation.h>
#include <equalizer/dictionary_templater.h>
#include <processor/post_caller.h>
#include <processor/url_encode.h>
#include <log/equalizer.h>

namespace yxiva { namespace equalizer {

typedef yplatform::future::future<void> future_send_result_t;

#define SENDER_CLOG(ctx, sev) YLOG_CTX_LOCAL(ctx, sev) << "[" << db_name_ << "-sender] "
#define SENDER_LOG(sev) YLOG_L(sev) << "[" << db_name_ << "-sender] "

template <typename HttpClient>
class notifications_sender_template
    : public yplatform::log::contains_logger
    , public std::enable_shared_from_this<notifications_sender_template<HttpClient>>
{
public:
    typedef std::function<void()> send_callback;

private:
    typedef boost::asio::io_service::strand strand_t;
    typedef notifications_sender_template<HttpClient> self_type;
    typedef std::pair<operation_ptr, send_callback> waiting_notification_info;
    typedef std::deque<waiting_notification_info> waiting_queue;

public:
    typedef std::shared_ptr<self_type> ptr;

    struct settings
    {
        string url;
        string auth_header;
        time_duration retry_interval = seconds(1);
        size_t user_queue_limit;
        size_t max_event_send_attempts;
    };

    notifications_sender_template(
        boost::asio::io_service& io,
        const settings& init_settings,
        const string& db_name,
        const std::shared_ptr<HttpClient>& http_client,
        const yplatform::log::source& logger = yplatform::log::source())
        : yplatform::log::contains_logger(logger)
        , ctx(make_shared<task_context>())
        , settings_(init_settings)
        , db_name_(url_encode(db_name))
        , strand_(io)
        , http_client_(http_client)
        , logger_(db_name_, yplatform::find<equalizer_log, std::shared_ptr>("eq_logger"))
        , running_(true)
    {
        retrier_ = std::make_shared<post_caller>(io, settings_.retry_interval);
    }

    void stop()
    {
        running_ = false;
        retrier_->stop();
        SENDER_LOG(info) << "stopped";
    }

    void send_notification(operation_ptr op, const send_callback& cb)
    {
        if (op->action_type == action_t::UNKNOWN)
        {
            drop(op, cb, "unknown_action");
        }
        else
        {
            op->ctx->profiler().extrude_and_push("wait_send");
            // This guarantees that there will be no concurrent requests for single user
            strand_.post(std::bind(
                &self_type::queue_or_send_immediately,
                this->shared_from_this(),
                op,
                std::move(cb)));
        }
    }

    const string& db_name() const
    {
        return db_name_;
    }

private:
    void queue_or_send_immediately(const operation_ptr& op, const send_callback& cb)
    {
        assert_strand();
        if (!running_)
        {
            return;
        }
        auto uid = op->uid();
        auto& queue = user_queues_[uid];
        bool request_in_progress = !queue.empty();
        enqueue(op, std::move(cb), queue);
        if (request_in_progress)
        {
            return;
        }
        send_queued(uid, queue);
    }

    void enqueue(const operation_ptr& op, const send_callback& cb, waiting_queue& queue)
    {
        if (queue.size() < settings_.user_queue_limit)
        {
            queue.push_back({ op, std::move(cb) });
        }
        else
        {
            drop(op, cb, "queue_overflow");
        }
    }

    void drop(const operation_ptr& op, const send_callback& cb, const string& reason)
    {
        op->ctx->profiler().pop();
        logger_.operation_dropped(op, reason);
        yplatform::safe_call(cb);
    }

    void send_queued(const string& uid)
    {
        assert_strand();
        if (!running_)
        {
            return;
        }
        auto& queue = user_queues_[uid];
        send_queued(uid, queue);
    }

    void send_queued(const string& uid, waiting_queue& queue)
    {
        drop_expired(queue);
        if (queue.empty())
        {
            user_queues_.erase(uid);
            return;
        }
        auto [payload, events_in_batch] = prepare_payload(uid, queue);
        auto req = yhttp::request::POST(
            settings_.url +
                yhttp::url_encode({ { "uid", uid }, { "events_in_batch", events_in_batch } }),
            "Content-Type: application/json\r\n" + settings_.auth_header,
            std::move(payload));
        namespace p = std::placeholders;
        http_client_->async_run(
            ctx,
            std::move(req),
            strand_.wrap(
                std::bind(&self_type::on_response, this->shared_from_this(), uid, p::_1, p::_2)));
    }

    void drop_expired(waiting_queue& queue)
    {
        while (queue.size() && queue.front().first->attempt >= settings_.max_event_send_attempts)
        {
            auto& [op, cb] = queue.front();
            drop(op, cb, "attempt_limit");
            queue.pop_front();
        }
    }

    auto prepare_payload(const string& uid, const waiting_queue& waiting_queue)
    {
        json_value task;
        task["uid"] = uid;
        task["suid"] = waiting_queue.front().first->suid();
        task["dbname"] = db_name_;
        auto&& events = task["events"];
        events.set_array();

        for (auto& [op, cb] : waiting_queue)
        {
            if (!op->attempt) op->ctx->profiler().extrude_and_push("send");
            ++op->attempt;
            json_value event = json_value(json_type::tobject);
            event["status"] = op->status;

            event["action"] = action_to_str[op->action_type];
            event["change_id"] = op->operation_id;
            event["args"] = op->args;
            event["ts"] = op->ts;
            event["lcn"] = op->lcn;
            event["request_id"] = op->x_request_id;

            auto&& items = event["items"];
            items.set_array();
            for (auto& item : op->parts)
            {
                items.push_back(item);
            }

            events.push_back(event);
        }

        return std::make_tuple(json_write(task), events.size());
    }

    void on_response(const string& uid, const boost::system::error_code& ec, yhttp::response resp)
    {
        assert_strand();
        if (!running_)
        {
            return;
        }
        if (ec || resp.status != 200)
        {
            log_http_error(ec, resp);
            retry(uid);
            return;
        }
        json_value response;
        if (auto res = json_parse(response, resp.body); !res)
        {
            SENDER_CLOG(ctx, error) << "invalid pusher response body: " << res.error_reason;
            retry(uid);
            return;
        }
        auto& queue = user_queues_[uid];
        auto processing_result = json_get<int>(response, "code", 0);
        if (processing_result == 205)
        {
            // Uninitialized user.
            ignore_operations(queue);
            user_queues_.erase(uid);
            return;
        }
        auto&& task = response["result"];
        if (!task.has_member("processed"))
        {
            SENDER_CLOG(ctx, error) << "missing processed count";
            retry(uid);
            return;
        }
        static constexpr auto bad_processed = std::numeric_limits<std::size_t>::max();
        auto processed = json_get(task, "processed", bad_processed);
        if (processed > queue.size())
        {
            SENDER_CLOG(ctx, error) << "invalid processed count: " << processed;
            retry(uid);
            return;
        }
        auto&& events = task["events"];
        acknowledge_sent(queue, processed);
        if (queue.empty())
        {
            user_queues_.erase(uid);
            return;
        }
        cache_metadata_for_unsent(queue, events);
        if (processing_result != 200)
        {
            SENDER_CLOG(ctx, error)
                << "error while sending notifications: response_code=" << processing_result
                << " error=\"" << response["error"] << "\"";
            retry(uid);
            return;
        }
        send_queued(uid, queue);
    }

    void log_http_error(const boost::system::error_code& ec, const yhttp::response& resp)
    {
        if (ec)
        {
            SENDER_CLOG(ctx, error) << "pusher request failed: \"" << ec.message() << "\"";
        }
        else
        {
            SENDER_CLOG(ctx, error) << "pusher request failed: response_code=" << resp.status;
        }
    }

    void retry(const string& uid)
    {
        retrier_->post(
            uid, strand_.wrap([uid, self = this->shared_from_this()] { self->send_queued(uid); }));
    }

    void ignore_operations(waiting_queue& queue)
    {
        for (auto& [op, cb] : queue)
        {
            logger_.operation_ignored(op, "user_not_initialized");
            op->ctx->profiler().pop();
            yplatform::safe_call(cb);
        }
        queue.clear();
    }

    void acknowledge_sent(waiting_queue& queue, std::size_t processed)
    {
        for (auto i = 0U; i < processed && queue.size(); ++i)
        {
            auto& [op, cb] = queue.front();
            logger_.operation_sent(op);
            op->ctx->profiler().pop();
            yplatform::safe_call(cb);
            queue.pop_front();
        }
    }

    void cache_metadata_for_unsent(waiting_queue& queue, json_value_ref& events)
    {
        for (auto i = 0U; i < std::min<size_t>(events.size(), queue.size()); ++i)
        {
            auto&& status = events[i]["status"];
            auto&& op = queue[i].first;

            if (!status.has_member("metadata_count"))
            {
                SENDER_CLOG(ctx, warning) << "ignoring pusher cache, missing meta_count";
                continue;
            }
            static constexpr auto bad_meta_count = std::numeric_limits<std::size_t>::max();
            auto meta_count = json_get(status, "metadata_count", bad_meta_count);
            if (meta_count > op->parts.size())
            {
                SENDER_CLOG(ctx, warning)
                    << "ignoring pusher cache, invalid meta_count: \"" << meta_count << "\"";
                continue;
            }
            op->status = status;
            auto&& items = events[i]["items"];
            for (auto j = 0U; j < std::min<size_t>(op->parts.size(), items.size()); ++j)
            {
                json_merge(op->parts[j], items[j]);
            }
        }
    }

    void assert_strand() const
    {
        assert(strand_.running_in_this_thread());
    }

private:
    task_context_ptr ctx;
    settings settings_;
    string db_name_;

    std::shared_ptr<post_caller> retrier_;
    strand_t strand_;

    std::unordered_map<string, waiting_queue> user_queues_;

    std::shared_ptr<HttpClient> http_client_;
    log_db_wrapper logger_;
    std::atomic_bool running_{ false };
};

#undef SENDER_CLOG
#undef SENDER_LOG

}}
