#include "settings.h"

#include <boost/assign/list_of.hpp>
#include <boost/unordered_map.hpp>

#include "util/read_ptree_to_collection.h"
#include "util/stream_settings_from_ptree.h"
#include "util/load_timeouts_from_ptree.h"
#include <yxiva/core/read_text_file.h>
#include <yplatform/time_traits.h>

namespace yxiva { namespace equalizer {

const boost::unordered_map<string, sync_dest> str_to_sync_dict =
    boost::assign::map_list_of("none", sync_dest::NONE)("file", sync_dest::FILE)(
        "distr_lock",
        sync_dest::DISTR_LOCK)("distr_file_lock", sync_dest::DISTR_FILE_LOCK);

namespace {
void validate_settings(const settings& st)
{
    if (st.stream.capacity <= 0) throw std::runtime_error("invalid default stream capacity");
    if (st.eq.stream.capacity <= 0) throw std::runtime_error("invalid equalizer stream capacity");
    if (st.eq.stream.window <= 1)
        throw std::runtime_error("invalid equalizer window (must be > 1)");
    if (st.eq.sender.url.empty()) throw std::runtime_error("equalizer target url is empty");
    if (st.selector.select_min_size > st.stream.capacity)
        throw std::runtime_error("selector min select size > stream capacity");
    if (st.selector.select_min_size > st.selector.select_max_size)
        throw std::runtime_error("selector min select size > max select size");
    if (st.auth.my_host_ip.empty()) throw std::runtime_error("can't retreive my ip address");
}

string lookup_my_host_ip()
{
    auto& io = *yplatform::global_net_reactor->io();
    using namespace boost::asio::ip;

    tcp::resolver resolver(io);
    tcp::resolver::query query(boost::asio::ip::host_name(), "");
    tcp::resolver::iterator it = resolver.resolve(query);
    tcp::endpoint ep = *it;

    return ep.address().to_string();
}

}

settings load_equalizer_settings(const yplatform::ptree& conf)
{
    settings st;
    stream_settings_from_ptree(conf, "", st.stream);

    st.use_selector = conf.get("use_local_selector", true);
    read_ptree(st.db_list, conf, "db_list");
    read_ptree(st.ignore_db_list, conf, "ignore_db_list");
    st.read_only_interval = conf.get("read_only_interval", st.read_only_interval);

    st.auth.cache_size = conf.get("auth.cache_size", 100U);
    st.auth.retry_interval = conf.get("auth.retry_interval", st.auth.retry_interval);
    stream_settings_from_ptree(conf, "auth", st.auth.stream, st.stream);

    st.pg_conn.sharpei_host = conf.get("pg.sharpei_host", "");
    st.pg_conn.username = conf.get("pg.user", "");
    st.pg_conn.poll_interval = conf.get("pg.poll_interval", st.pg_conn.poll_interval);

    st.aggregator = conf.get("aggregator", "xiva");
    stream_settings_from_ptree(conf, "equalizer", st.eq.stream, st.stream);
    st.eq.sender.url = conf.get<string>("equalizer.host") + conf.get("equalizer.url", "");
    if (auto http = conf.get_child_optional("http"))
    {
        st.http.parse_ptree(http.get());
    }
    if (auto auth = conf.get_child_optional("equalizer.auth"))
    {
        st.eq.sender.auth_header = "Authorization: " + auth->get<string>("type") + " " +
            read_text_file(auth->get<string>("credentials_file")) + "\r\n";
    }
    st.eq.sender.retry_interval = conf.get("equalizer.retry_interval", st.eq.sender.retry_interval);
    st.eq.sender.user_queue_limit = conf.get("equalizer.user_queue_limit", 100UL);
    st.eq.sender.max_event_send_attempts = conf.get("equalizer.max_event_send_attempts", 3);

    st.selector.select_min_size = conf.get("selector.select_min_size", 10U);
    st.selector.select_max_size = conf.get("selector.select_max_size", 10U);
    st.selector.correct_start_position = conf.get("selector.correct_start_position", true);
    st.selector.enable_trace_notification = conf.get("selector.trace", true);
    st.selector.max_offset_minutes = conf.get("selector.max_offset_minutes", 0U);
    read_ptree(st.selector.retry_intervals, conf.get_child_optional("selector"), "retry_interval");

    string sync_to_str = conf.get("sync.dest", "none");
    auto sync_it = str_to_sync_dict.find(sync_to_str);
    if (sync_it == str_to_sync_dict.end()) throw std::runtime_error("unknown sync destination");
    st.syncer.dest = sync_it->second;
    st.syncer.path = conf.get("sync.path", "./");

    st.auth.my_host_ip = lookup_my_host_ip();
    YLOG_G(info) << "my host ip: ip=\"" << st.auth.my_host_ip << "\"";

    validate_settings(st);
    return st;
}

}}
