#pragma once

#include <map>
#include <mutex>
#include <cctype>
#include <algorithm>

#include <yplatform/find.h>
#include <ymod_httpclient/call.h>
#include <ymod_httpclient/response_handler.h>
#include <yxiva/core/http_handlers.h>
#include <yxiva/core/json.h>
#include <equalizer/context.h>

namespace yxiva { namespace equalizer {

class pg_conninfo_provider : public std::enable_shared_from_this<pg_conninfo_provider>
{
    typedef http_handler_read_body handler_t;
    typedef boost::shared_ptr<handler_t> handler_ptr;

    struct db_info
    {
        string conninfo;
        uint64_t lag;
        string alias;

        bool operator==(const db_info& other) const
        {
            return std::tie(conninfo, lag, alias) ==
                std::tie(other.conninfo, other.lag, other.alias);
        }
    };

    typedef std::map<string, db_info> db_map;

public:
    struct db_identity
    {
        string id;
        string alias;
        db_identity(const string& id, const string& alias) : id(id), alias(alias)
        {
        }
    };
    typedef std::function<void(void)> db_map_updated_callback;

    struct settings
    {
        string sharpei_host;
        string username;
        time_duration poll_interval = seconds(30);
    };

    pg_conninfo_provider(
        const settings& st,
        boost::asio::io_service& io,
        db_map_updated_callback&& cb)
        : settings_(st)
        , timer_(io)
        , stopped_(true)
        , update_cb_(static_cast<db_map_updated_callback&&>(cb))
    {
        if (!st.sharpei_host.empty())
        {
            http_client_ = yplatform::find<yhttp::call>("http_client");
            host_info_ = http_client_->make_rm_info(st.sharpei_host);
        }
    }

    void start()
    {
        scoped_lock guard(mutex_);
        if (!host_info_)
        {
            YLOG_G(error)
                << "[pg_conninfo_provider] lookup_conninfos failed: no sharpei host provided";
            return;
        }

        if (!stopped_) return;
        stopped_ = false;
        guard.unlock();
        lookup_conninfos();
    }

    void reload(const settings& st)
    {
        scoped_lock guark(mutex_);
        settings_ = st;
    }

    void stop()
    {
        scoped_lock guard(mutex_);
        if (stopped_) return;
        stopped_ = true;
        timer_.cancel();
    }

    const string conninfo(const string& db_name) const
    {
        scoped_lock guard(mutex_);
        auto iconninfo = db_infos_.find(db_name);
        if (iconninfo != db_infos_.end())
            return iconninfo->second.conninfo + " user=" + settings_.username +
                " sslmode=verify-full";
        return "";
    }

    const string alias(const string& db_name) const
    {
        scoped_lock guard(mutex_);
        auto iconninfo = db_infos_.find(db_name);
        if (iconninfo != db_infos_.end()) return iconninfo->second.alias;
        return "";
    }

    uint64_t database_lag(const string& db_name) const
    {
        scoped_lock guard(mutex_);
        auto iconninfo = db_infos_.find(db_name);
        if (iconninfo != db_infos_.end()) return iconninfo->second.lag;
        return 0U;
    }

    const std::vector<db_identity> all_databases() const
    {
        scoped_lock guard(mutex_);
        std::vector<db_identity> names;
        names.reserve(db_infos_.size());
        for (auto& conninfo : db_infos_)
        {
            names.emplace_back(conninfo.first, conninfo.second.alias);
        }
        guard.unlock();
        return names;
    }

protected:
    void lookup_conninfos(const boost::system::error_code& err = boost::system::error_code())
    {
        if (err == boost::asio::error::operation_aborted)
        {
            return;
        }

        auto handler = boost::make_shared<http_handler_read_body>();
        auto ctx = boost::make_shared<context>();

        try
        {
            auto response = http_client_->get_url(ctx, handler, host_info_, "v2/stat");
            response.add_callback(boost::bind(
                &pg_conninfo_provider::on_response,
                this->shared_from_this(),
                response,
                handler,
                ctx));
        }
        catch (const std::exception& ex)
        {
            assert(false && "Prepare http call not supposed to throw exceptions");
        }
        catch (...)
        {
            assert(false && "Prepare http call not supposed to throw exceptions");
        }
    }

    void on_response(ymod_http_client::future_void_t response, handler_ptr handler, context_ptr ctx)
    {
        try
        {
            response.get();
            if (handler->success())
            {
                process_response(handler->data(), ctx);
            }
        }
        catch (const std::exception& ex)
        {
            YLOG_CTX_GLOBAL(ctx, error)
                << "[pg_conninfo_provider] error on response: exception=\"" << ex.what() << "\"";
        }
        catch (...)
        {
            YLOG_CTX_GLOBAL(ctx, error)
                << "[pg_conninfo_provider] error on response: exception=\"unknown\"";
        }
        repeat_lookup();
    }

    void repeat_lookup()
    {
        scoped_lock guard(mutex_);
        if (stopped_) return;
        timer_.expires_from_now(settings_.poll_interval);
        timer_.async_wait(
            boost::bind(&pg_conninfo_provider::lookup_conninfos, shared_from_this(), _1));
    }

    void process_response(const string& data, context_ptr ctx)
    {
        static const char* prefix = "xdb";
        json_value db_infos = json_parse(data);
        db_map updated_db_infos;

        for (auto item = db_infos.members_begin(); item != db_infos.members_end(); ++item)
        {
            string key(item.key());
            auto&& value = *item;

            if (value.type() != json_type::tobject) continue;

            boost::optional<json_value> database =
                find_alive_database(value["databases"], "replica");
            if (!database) database = find_alive_database(value["databases"]);

            if (database)
            {
                auto dbinfo = get_dbinfo(database.get());
                string db_id = json_get<string>(value, "id", "");
                if (dbinfo && db_id.size())
                {
                    string db_alias = prefix + db_id;
                    YLOG_CTX_GLOBAL(ctx, debug) << "retrieved dbinfo: db_name=\"" << db_alias
                                                << "\" "
                                                   "conninfo=\""
                                                << dbinfo.get().conninfo
                                                << "\" "
                                                   "alias=\""
                                                << dbinfo.get().alias << "\"";
                    dbinfo.get().alias = json_get<string>(value, "name", "");
                    updated_db_infos[db_alias] = dbinfo.get();
                }
            }
        }
        {
            scoped_lock guard(mutex_);
            if (updated_db_infos != db_infos_)
            {
                std::swap(updated_db_infos, db_infos_);
                guard.unlock();
                update_cb_();
            }
        }
    }

    boost::optional<json_value> find_alive_database(
        const json_value_ref& hosts_array,
        const string& role = "")
    {
        if (hosts_array.type() == json_type::tarray)
        {
            for (auto&& node : hosts_array.array_items())
            {
                if (!node.empty() &&
                    (role == "" || json_decoder<string>::get(node["role"], "") == role) &&
                    json_decoder<string>::get(node["status"], "dead") == "alive")
                {
                    return json_value(node);
                }
            }
        }
        return boost::optional<json_value>();
    }

    boost::optional<db_info> get_dbinfo(const json_value_ref& node)
    {
        if (node.type() == json_type::tobject)
        {
            json_value address = node["address"];
            if (address.type() == json_type::tobject)
            {
                string host = json_decoder<string>::get(address["host"], "");
                string port = json_decoder<string>::get(address["port"], "");
                string dbname = json_decoder<string>::get(address["dbname"], "");

                auto conninfo = string("host=") + host + " port=" + port + " dbname=" + dbname;
                auto lag = json_get<uint64_t>(node["state"], "lag", 0U);
                if (!host.empty() && !port.empty() && !dbname.empty())
                {
                    return db_info{ conninfo, lag, "" };
                }
            }
        }
        return boost::optional<db_info>();
    }

private:
    ymod_http_client::remote_point_info_ptr host_info_;
    boost::shared_ptr<yhttp::call> http_client_;
    settings settings_;
    time_traits::timer timer_;

    mutable mutex mutex_;
    db_map db_infos_;
    bool stopped_;
    db_map_updated_callback update_cb_;
};

typedef std::shared_ptr<pg_conninfo_provider> conninfo_provider_ptr;
}}
