#pragma once

#include <ctime>
#include <boost/algorithm/string/trim.hpp>

#include <ymod_pq/call.h>
#include <yxiva/core/json.h>

#include <equalizer/operation.h>
#include <generator/pg_action_type.h>

namespace yxiva { namespace equalizer {

class pg_operations_handler : public ymod_pq::response_handler
{
    enum columns
    {
        OPERATION_ID = 0,    // 0
        UID,                 // 1
        REVISION,            // 2
        ACTION_TYPE,         // 3
        ACTION_ARGUMENTS,    // 4
        MAILS_CHANGED_STATE, // 5
        FRESH_COUNT,         // 6
        USEFUL_NEW_COUNT,    // 7
        SESSION_KEY,         // 8
        OPERATION_DATE,      // 9
        X_REQUEST_ID,        // 10

        COLUMN_COUNT
    };

    std::vector<const char*> column_name = {
        "OPERATION_ID",        "UID",         "REVISION",         "ACTION_TYPE", "ACTION_ARGUMENTS",
        "MAILS_CHANGED_STATE", "FRESH_COUNT", "USEFUL_NEW_COUNT", "SESSION_KEY", "OPERATION_DATE",
        "X_REQUEST_ID",
    };

public:
    typedef std::vector<operation_ptr> operations_list_t;
    typedef std::shared_ptr<operations_list_t> operations_list_ptr;

    pg_operations_handler(std::size_t reserve_size)
        : operations_list_(std::make_shared<operations_list_t>())
    {
        operations_list_->reserve(reserve_size);
    }

    virtual ~pg_operations_handler()
    {
    }

    void handle_cell(unsigned /*row*/, unsigned col, const string& val, bool is_null)
    {
        if (col >= COLUMN_COUNT)
        {
            YLOG_G(warning) << "unmapped column #" << col << " value=\"" << val << "\"";
            return;
        }

        string value = is_null ? "" : val;
        try
        {
            fill_operation(col, value, is_null);
        }
        catch (const boost::bad_lexical_cast& ex)
        {
            YLOG_CTX_GLOBAL(operation_->ctx, error) << "bad cast error: column=" << column_name[col]
                                                    << " exception=\"" << ex.what() << "\""
                                                    << " value=\"" << value << "\"";
        }
        catch (const std::exception& ex)
        {
            YLOG_CTX_GLOBAL(operation_->ctx, error)
                << "error on select: column=" << column_name[col] << " exception=\"" << ex.what()
                << "\""
                << " value=\"" << value << "\"";
        }
    }

    void fill_operation(unsigned col, string& value, bool is_null)
    {
        switch (col)
        {
        case OPERATION_ID:
            operation_->operation_id = boost::lexical_cast<std::size_t>(value);
            break;
        case UID:
            boost::trim(value);
            operation_->ui.uid = value;
            operation_->ui.mdb = "pg";
            break;
        case REVISION:
            boost::trim(value);
            operation_->lcn = value;
            break;
        case ACTION_TYPE:
            boost::trim(value);
            raw_action_type_ = value;
            break;
        case ACTION_ARGUMENTS:
            if (!value.empty()) json_merge(operation_->args, json_parse(value));
            break;
        case MAILS_CHANGED_STATE:
            if (!value.empty()) fill_parts(value);
            break;
        case FRESH_COUNT:
            operation_->args["fresh_count"] = is_null ? 0 : boost::lexical_cast<int>(value);
            break;
        case USEFUL_NEW_COUNT:
            operation_->args["useful_new_messages"] = is_null ? 0 : boost::lexical_cast<int>(value);
            break;
        case SESSION_KEY:
            if (!is_null)
            {
                boost::trim(value);
                operation_->args["session_key"] = value;
            }
            break;
        case X_REQUEST_ID:
            if (!is_null)
            {
                boost::trim(value);
                operation_->x_request_id = value;
            }
            break;
        case OPERATION_DATE:
            operation_->ts = is_null ? 0 : boost::lexical_cast<std::time_t>(value);
            break;
        default:
            break;
        }
    }

    void fill_parts(const string& value)
    {
        json_value mail_states = json_parse(value, json_type::tarray);
        if (!mail_states.empty())
        {
            for (auto&& mstate : mail_states.array_items())
            {
                if (mstate.type() != json_type::tobject)
                {
                    YLOG_CTX_GLOBAL(operation_->ctx, error)
                        << "invalid mail state in changed field (not json object)";
                }
                else
                {
                    operation_->parts.push_back(mstate);
                }
            }
            operation_->total_count = operation_->parts.size();
        }
    }

    unsigned column_count() const
    {
        return COLUMN_COUNT;
    }

    void handle_row_begin(unsigned /*row*/)
    {
        raw_action_type_.clear();
        operation_ = std::make_shared<operation>();
        operation_->ctx->profiler().push("select_output");
    }

    void handle_row_end(unsigned /*row*/)
    {
        operation_->action_type = resolve_action_type(raw_action_type_, operation_->args);
        if (operation_->action_type == action_t::MARK_MAILS)
        {
            fill_labels("lids_add");
        }
        else if (operation_->action_type == action_t::UNMARK_MAILS)
        {
            fill_labels("lids_del");
        }
        else if (operation_->action_type == action_t::TRANSFER)
        {
            operation_->ui.suid = json_get(operation_->args, "suid", string());
        }
        operations_list_->push_back(operation_);
        operation_.reset();
    }

    void fill_labels(const char* key) const
    {
        auto&& labels = operation_->args["labels"];
        labels.set_array();
        auto&& array = operation_->args[key];
        if (array.type() == json_type::tarray)
        {
            for (size_t i = 0; i < array.size(); i++)
            {
                labels.push_back(json_decoder<string>::get(array[i], ""));
            }
        }
    }

    operations_list_ptr list() const
    {
        return operations_list_;
    }

private:
    string raw_action_type_;
    operation_ptr operation_;
    operations_list_ptr operations_list_;
};

}}