#pragma once

#include <ymod_pq/call.h>
#include <yplatform/future/future.hpp>
#include <yplatform/find.h>
#include <yplatform/task_context.h>

#include <equalizer/context.h>
#include <equalizer/operation.h>
#include <processor/pipeline_types.h>
#include "pg_conninfo_provider.h"
#include "pg_operations_handler.h"
#include "position_handler.h"
#include "selector.h"

namespace yxiva { namespace equalizer {

class pg_select_adaptor : public idb_select_adaptor<operation_ptr>
{
    typedef boost::shared_ptr<pg_operations_handler> operations_handler_ptr;
    typedef boost::shared_ptr<position_handler> position_handler_ptr;
    typedef ymod_pq::future_result future_result;
    typedef boost::shared_ptr<ymod_pq::call> call_ptr;

protected:
    typedef idb_select_adaptor<operation_ptr> base_t;

public:
    pg_select_adaptor(conninfo_provider_ptr conninfo_provider)
        : conninfo_provider_(conninfo_provider)
    {
        db_call_ = yplatform::find<ymod_pq::call>("pq");
        assert(db_call_);
    }

protected:
    position_future_t select_start_position(
        std::size_t max_offset_minutes,
        const string& db_name,
        context_ptr ctx)
    {
        string conn = conninfo(db_name);

        position_handler_ptr handler = boost::make_shared<position_handler>(ctx);
        ymod_pq::bind_array_ptr args(new ymod_pq::bind_array);
        ymod_pq::push_const_uint(args, static_cast<uint32_t>(max_offset_minutes));

        future_result res = db_call_->request(ctx, conn, "start_position", args, handler, true);
        position_promise_t prom;
        res.add_callback(boost::bind(
            &pg_select_adaptor::on_select_start_position,
            get_shared_from_this(),
            res,
            handler,
            prom));
        return prom;
    }

    void on_select_start_position(
        future_result res,
        position_handler_ptr handler,
        position_promise_t prom) noexcept
    {
        try
        {
            res.get();
            prom.set(handler->max_operation_id());
        }
        catch (...)
        {
            prom.set_current_exception();
        }
    }

    operations_future_t select_operations(
        std::size_t start_id,
        std::size_t select_size,
        const string& db_name,
        context_ptr ctx)
    {
        string conn = conninfo(db_name);

        ymod_pq::bind_array_ptr args(new ymod_pq::bind_array);
        ymod_pq::push_const_string(args, std::to_string(start_id));
        ymod_pq::push_const_uint(args, static_cast<uint32_t>(select_size));

        operations_handler_ptr handler = boost::make_shared<pg_operations_handler>(select_size);

        auto res = db_call_->request(ctx, conn, "list_operations", args, handler, true);
        operations_promise_t prom;
        res.add_callback(boost::bind(
            &pg_select_adaptor::on_select_operations, get_shared_from_this(), res, handler, prom));
        return prom;
    }

    void on_select_operations(
        future_result res,
        operations_handler_ptr handler,
        operations_promise_t prom) noexcept
    {
        try
        {
            res.get();
            prom.set(handler->list());
        }
        catch (...)
        {
            prom.set_current_exception();
        }
    }

    uint64_t current_db_lag(const string& db_name) const
    {
        return conninfo_provider_->database_lag(db_name);
    }

    string conninfo(const string& db_name)
    {
        string conn = conninfo_provider_->conninfo(db_name);
        if (conn.empty()) throw std::runtime_error("no conninfo provided");
        return conn;
    }

private:
    std::shared_ptr<pg_select_adaptor> get_shared_from_this()
    {
        return std::dynamic_pointer_cast<pg_select_adaptor>(this->shared_from_this());
    }

private:
    call_ptr db_call_;
    conninfo_provider_ptr conninfo_provider_;
};

}}
