#pragma once

#include <equalizer/context.h>
#include <processor/pipeline_types.h>
#include "selector_settings.h"
#include "idb_select_adaptor.h"

namespace yxiva { namespace equalizer {

using namespace pipeline;

#define SELECTOR_LOG(ctx, severity) YLOG_CTX_LOCAL(ctx, severity) << "[" << mdb_ << "-selector] "

class selector
    : public Generator<StreamStrand<operation_ptr>>
    , public yplatform::log::contains_logger
{
protected:
    typedef Generator<StreamStrand<operation_ptr>> base_t;
    typedef selector this_t;

    typedef idb_select_adaptor<operation_ptr> db_adaptor;
    typedef typename db_adaptor::ptr db_adaptor_ptr;
    typedef typename db_adaptor::operations_list_t operations_list_t;
    typedef typename db_adaptor::operations_list_ptr operations_list_ptr;
    typedef typename db_adaptor::position_future_t position_future_t;
    typedef typename db_adaptor::operations_future_t operations_future_t;

public:
    selector(
        boost::asio::io_service& io,
        const selector_settings& init_settings,
        string mdb,
        std::size_t start_operation_id,
        db_adaptor_ptr adaptor,
        const yplatform::log::source& logger = yplatform::log::source())
        : base_t(io, init_settings.select_min_size, init_settings.retry_intervals)
        , yplatform::log::contains_logger(logger)
        , settings_(init_settings)
        , start_operation_id_(start_operation_id)
        , is_start_position_corrected_(!init_settings.correct_start_position)
        , retrying_sequence_max_id_(0)
        , is_retry_broken_started_(false)
        , mdb_(mdb)
        , ctx_(boost::make_shared<context>())
        , adaptor_(adaptor)
    {
        if (mdb_.substr(0, 3) == "xdb")
            ctx_ = boost::make_shared<context>(ctx_->uniq_id() + "_" + mdb_);
    }

    void pause()
    {
        SELECTOR_LOG(ctx_, info) << " paused";
        base_t::pause();
    }

    void resume()
    {
        SELECTOR_LOG(ctx_, info) << " resumed";
        base_t::resume();
    }

    void start()
    {
        SELECTOR_LOG(ctx_, info) << "started: position=" << start_operation_id_;
        base_t::start();
    }

    void stop()
    {
        SELECTOR_LOG(ctx_, info) << " stopped";
        base_t::stop();
    }

    std::size_t start_select_position() const
    {
        return start_operation_id_;
    }

    std::size_t retrying_sequence_max_id() const
    {
        return retrying_sequence_max_id_;
    }

    const selector_settings& settings() const
    {
        return settings_;
    }

private:
    void generate(std::size_t free_space)
    {
        if (!is_start_position_corrected_)
        {
            correct_start_position(free_space);
            return;
        }

        std::size_t select_size = std::min(free_space, settings_.select_max_size);

        SELECTOR_LOG(ctx_, debug) << "start select: free_space=" << free_space
                                  << " start_operation_id=" << start_operation_id_
                                  << " select_size=" << select_size;

        try
        {
            auto operations_request =
                adaptor_->select_operations(start_operation_id_, select_size, mdb_, ctx_);
            operations_request.add_callback(
                boost::bind(&selector::on_select, get_shared_from_this(), operations_request));
            return;
        }
        catch (const std::exception& ex)
        {
            SELECTOR_LOG(ctx_, error)
                << "select operations request exception=\"" << ex.what() << "\"";
        }
        catch (...)
        {
            SELECTOR_LOG(ctx_, error) << "select operations request exception=\"unknown\"";
        }
        this->generate_failed();
    }

    void correct_start_position(std::size_t free_space)
    {
        try
        {
            auto start_position_request =
                adaptor_->select_start_position(settings_.max_offset_minutes, mdb_, ctx_);
            start_position_request.add_callback(boost::bind(
                &selector::on_fix_position,
                get_shared_from_this(),
                start_position_request,
                free_space));
            return;
        }
        catch (const std::exception& ex)
        {
            SELECTOR_LOG(ctx_, error)
                << "select start position request exception=\"" << ex.what() << "\"";
        }
        catch (...)
        {
            SELECTOR_LOG(ctx_, error) << "select start position request exception=\"unknown\"";
        }
        this->generate_failed();
    }

    void on_fix_position(position_future_t position, std::size_t free_space)
    {
        try
        {
            start_operation_id_ = std::max(start_operation_id_, position.get());
            SELECTOR_LOG(ctx_, info) << "start position updated: position=" << start_operation_id_;
            is_start_position_corrected_ = true;
            generate(free_space);
            return;
        }
        catch (const std::exception& ex)
        {
            SELECTOR_LOG(ctx_, error)
                << "select start position result exception=\"" << ex.what() << "\"";
        }
        catch (...)
        {
            SELECTOR_LOG(ctx_, error) << "select start position result exception=\"unknown\"";
        }
        this->generate_failed();
    }

    void on_select(operations_future_t operations)
    {
        try
        {
            process_operations(operations.get());
        }
        catch (const std::exception& ex)
        {
            SELECTOR_LOG(ctx_, error)
                << "select operations result exception=\"" << ex.what() << "\"";
            this->generate_failed();
        }
        catch (...)
        {
            SELECTOR_LOG(ctx_, error) << "select operations result exception=\"unknown\"";
            this->generate_failed();
        }
    }

    void process_operations(operations_list_ptr operations)
    {
        SELECTOR_LOG(ctx_, debug) << "selected: count=" << operations->size();
        if (auto processed_list = fix_put_sequence(operations))
        {
            bool trace_sequence = false;
            if (!processed_list->empty())
            {
                update_start_position(processed_list->back()->operation_id);
            }
            if (settings_.enable_trace_notification && operations->empty())
            {
                auto trace_operation = std::make_shared<operation>();
                trace_operation->operation_id = start_operation_id_;
                trace_operation->action_type = action_t::TRACE_PIPELINE;
                trace_operation->ts -= adaptor_->current_db_lag(mdb_);
                trace_operation->ctx->profiler().push("select_output");
                processed_list->push_back(trace_operation);
                trace_sequence = true;
            }
            this->on_generated(processed_list, trace_sequence);
        }
        else
        {
            this->generate_failed();
        }
    }

protected:
    void update_start_position(std::size_t operation_id)
    {
        SELECTOR_LOG(ctx_, info) << "updating start_operation_id: old=" << start_operation_id_
                                 << " new=" << operation_id;
        start_operation_id_ = operation_id;
    }

    operations_list_ptr fix_put_sequence(operations_list_ptr operations)
    {
        if (operations->empty()) return operations;

        if (!is_retry_broken_started_) retrying_sequence_max_id_ = operations->back()->operation_id;

        std::size_t unretried_sequence_begin = start_operation_id_;
        if (this->is_last_retry_interval()) unretried_sequence_begin = retrying_sequence_max_id_;

        auto continuous_sequence_end =
            find_continuous_leading_sequence_end(operations, unretried_sequence_begin);
        if (continuous_sequence_end != operations->begin())
        {
            is_retry_broken_started_ = false;
            return cut_operations(operations, continuous_sequence_end);
        }

        if (!is_retry_broken_started_)
        {
            is_retry_broken_started_ = true;
            this->reset_retry_intervals();
        }

        SELECTOR_LOG(ctx_, info) << "sequence broken, retry needed: max_operation_id="
                                 << retrying_sequence_max_id_;

        assert(!this->is_last_retry_interval());
        return nullptr;
    }

private:
    typename operations_list_t::iterator find_continuous_leading_sequence_end(
        operations_list_ptr operations,
        std::size_t unretried_sequence_begin) const
    {
        id_t previous_id = start_operation_id_;
        for (auto it = operations->begin(); it != operations->end(); it++)
        {
            auto operation = *it;
            if (operation->operation_id - 1 != previous_id && previous_id > 0)
            {
                if (operation->operation_id > unretried_sequence_begin)
                {
                    return it;
                }
                else
                {
                    SELECTOR_LOG(ctx_, warning)
                        << "broken operation_id sequence:"
                           " previous="
                        << previous_id << " next=" << operation->operation_id;
                }
            }
            previous_id = operation->operation_id;
        }
        return operations->end();
    }

    static operations_list_ptr cut_operations(
        operations_list_ptr operations,
        typename operations_list_t::iterator end)
    {
        if (end == operations->end())
        {
            return operations;
        }
        else
        {
            operations_list_ptr suboperations;
            suboperations = std::make_shared<operations_list_t>(operations->begin(), end);
            assert(!suboperations->empty());
            return suboperations;
        }
    }

    std::shared_ptr<this_t> get_shared_from_this()
    {
        return std::dynamic_pointer_cast<this_t>(this->shared_from_this());
    }

private:
    const selector_settings settings_;
    std::size_t start_operation_id_;
    bool is_start_position_corrected_;

    std::size_t retrying_sequence_max_id_;
    bool is_retry_broken_started_;

    const string mdb_;
    context_ptr ctx_;
    db_adaptor_ptr adaptor_;
};

}}
