#pragma once

#include <mutex>

#include <boost/function.hpp>
#include <boost/asio.hpp>

#include <yplatform/find.h>
#include <yplatform/log.h>
#include <ymod_lease/ymod_lease.h>
#include <ymod_lease/lock_manager.h>

#include "types.h"

namespace yxiva { namespace equalizer {

struct lock_manager_settings
{
    std::size_t max_owned_locks_count = 1024U;
    std::size_t extra_acquire_count = 10U;
};

// TODO merge with lock_manager_module
class lock_manager : public yplatform::module
{
    typedef ylease::lock_manager<ylease::node> implementetion_type;

public:
    void bind(lock_callback_t&& on_lock, lock_callback_t&& on_unlock, update_callback_t&& on_update)
    {
        assert(!impl_);
        lock_manager_settings st;
        {
            scoped_lock lock(mutex_);
            st = settings_;
        }
        impl_ = std::make_shared<implementetion_type>(
            io_, lease_node_, st.max_owned_locks_count, st.extra_acquire_count);
        impl_->init(std::move(on_lock), std::move(on_unlock));
        impl_->subscribe_for_value_update(std::move(on_update));
        impl_->logger(this->logger());
    }

    void lock(const std::vector<std::string>& names)
    {
        assert(impl_);
        impl_->on_add_resources(names);
    }

    void unlock(const std::vector<std::string>& names)
    {
        assert(impl_);
        impl_->on_del_resources(names);
    }

    void read_only(const std::vector<std::string>& names, const time_duration& duration)
    {
        assert(impl_);
        impl_->release_resources_for(names, duration);
    }

    void lock(const string& name)
    {
        assert(impl_);
        impl_->on_add_resources({ name });
    }

    void unlock(const string& name)
    {
        assert(impl_);
        impl_->on_del_resources({ name });
    }

    void read_only(const string& name, const time_duration& duration)
    {
        assert(impl_);
        impl_->release_resources_for({ name }, duration);
    }

    void get_owned_locks(const std::function<void(std::vector<std::string>)>& cb) const
    {
        return impl_->get_acquired_resources(cb);
    }

protected:
    void init(boost::asio::io_service* io, lock_manager_settings st)
    {
        io_ = io;
        settings_ = st;
        try
        {
            lease_node_ = yplatform::find<ylease::node, std::shared_ptr>("lease-node");
            lease_node_->subscribe_peers_count(
                std::bind(&lock_manager::update_clients_count, this, std::placeholders::_1));
        }
        catch (const std::exception& ex)
        {
            YLOG_L(error) << "init: exception=\"" << ex.what() << "\"";
        }
    }

    void fini()
    {
        impl_.reset();
    }

private:
    void update_clients_count(std::size_t count)
    {
        impl_->update_peers_count(count < 1U ? 1U : static_cast<unsigned>(count));
    }

private:
    std::shared_ptr<implementetion_type> impl_;
    boost::asio::io_service* io_;
    std::shared_ptr<ylease::node> lease_node_;
    lock_manager_settings settings_;
    mutable mutex mutex_;
};

}}
