#pragma once

#include <boost/bind.hpp>

#include <yxiva/core/types.h>
#include <yplatform/reactor.h>

namespace yxiva { namespace equalizer {

template <typename PositionType = unsigned long>
class position_holder : public std::enable_shared_from_this<position_holder<PositionType>>
{
    typedef position_holder<PositionType> this_t;

public:
    class isync_strategy
    {
    public:
        virtual ~isync_strategy(){};
        virtual PositionType restore(const string& resource_name) = 0;
        virtual void sync(const string& resource_name, const PositionType& value) = 0;
    };

    typedef std::shared_ptr<isync_strategy> sync_strategy_ptr;

    position_holder(sync_strategy_ptr syncer) : syncer_(syncer)
    {
        assert(syncer);
    }

    virtual ~position_holder()
    {
    }

    PositionType position(const string& resource_name)
    {
        scoped_lock guard(mutex_);
        auto it = positions_.find(resource_name);
        if (it == positions_.end())
        {
            guard.unlock();
            return restore(resource_name);
        }
        return it->second;
    }

    void position(const string& resource_name, const PositionType& position)
    {
        scoped_lock guard(mutex_);
        PositionType sync_position = position;
        auto iposition = positions_.find(resource_name);
        if (iposition == positions_.end())
        {
            positions_.insert(iposition, std::make_pair(resource_name, position));
        }
        else
        {
            if (iposition->second == position)
            {
                // value already stored
                return;
            }
            else if (iposition->second > position)
            {
                // decrease is not allowed
                return;
            }
            else
            {
                iposition->second = position;
            }
        }
        guard.unlock();

        auto& io = *yplatform::global_net_reactor->io();
        io.post(boost::bind(
            &this_t::make_sync, this->shared_from_this(), resource_name, sync_position));
    }

    void update(const string& resource_name, const string& value)
    {
        if (!value.empty())
        {
            PositionType casted = boost::lexical_cast<PositionType>(value);
            position(resource_name, casted);
        }
    }

    void reset_syncer(sync_strategy_ptr syncer)
    {
        assert(syncer);
        scoped_lock guard(mutex_);
        syncer_ = syncer;
        auto positions_copy = positions_;
        guard.unlock();

        for (auto& position : positions_copy)
        {
            syncer->sync(position.first, position.second);
        }
    }

private:
    PositionType restore(const string& resource_name)
    {
        scoped_lock guard(mutex_);
        auto syncer = syncer_;
        guard.unlock();

        PositionType restored = syncer->restore(resource_name);

        guard.lock();
        if (!positions_.count(resource_name) || positions_[resource_name] < restored)
        {
            positions_[resource_name] = restored;
        }

        return positions_[resource_name];
    }

    void make_sync(const string& resource_name, PositionType sync_position)
    {
        scoped_lock guard(mutex_);
        if (auto syncer = syncer_)
        {
            guard.unlock();
            try
            {
                YLOG_G(debug) << "[position_holder] updating position: resource=\"" << resource_name
                              << "\" value=" << sync_position;
                syncer->sync(resource_name, sync_position);
            }
            catch (const std::exception& ex)
            {
                YLOG_L(error) << "[position_holder] error while backuping position: resourse=\""
                              << resource_name << "\" error=\"" << ex.what() << "\"";
            }
        }
    }

private:
    std::map<string, PositionType> positions_;
    sync_strategy_ptr syncer_;

    mutex mutex_;
};

}}