#pragma once

#include <fstream>

#include <boost/bind.hpp>
#include <boost/enable_shared_from_this.hpp>
#include <boost/filesystem.hpp>
#include <boost/lexical_cast.hpp>

#include "position_holder.h"

namespace yxiva { namespace equalizer {

template <typename PositionType = unsigned long>
class sync_file : virtual public position_holder<PositionType>::isync_strategy
{
    typedef std::map<string, boost::shared_ptr<std::ofstream>> backups_map;

public:
    sync_file(const string& basepath, const PositionType& def) : basepath_(basepath), default_(def)
    {
        boost::filesystem::path base(basepath_);
        if (base.is_relative())
        {
            base = boost::filesystem::absolute(base);
        }
        boost::filesystem::create_directories(base);
    }

    PositionType restore(const string& resource_name)
    {
        PositionType restored_value;
        std::ifstream backup;
        string filename = get_filename(resource_name);
        backup.open(filename);
        if (backup.is_open())
        {
            PositionType value = 0;
            string str_value;
            backup >> str_value;
            try
            {
                value = boost::lexical_cast<PositionType>(str_value);
            }
            catch (const std::bad_cast& ex)
            {
                YLOG_G(error) << "[sync_file] file with lock contains bad value:"
                                 " resource=\""
                              << resource_name
                              << "\""
                                 " file=\""
                              << filename
                              << "\""
                                 " value=\""
                              << str_value << "\"";
            }
            YLOG_G(info) << "[sync_file] value restored from file: name=\"" << resource_name
                         << "\""
                            " value="
                         << value << "";
            restored_value = value;
        }
        else
        {
            restored_value = default_;
        }

        return restored_value;
    }

    void sync(const string& resource_name, const PositionType& value)
    {
        scoped_lock guard(mutex_);
        auto ibackup = prepare_output(resource_name);
        if (ibackup == backups_.end())
        {
            YLOG_G(error) << "[sync_file] backup not created: " << value;
            return;
        }
        auto file = ibackup->second;
        file->seekp(0);
        *file << value << "\n";
        file->flush();
    }

protected:
    inline string get_filename(const string& resource_name)
    {
        return basepath_ + resource_name + ".lock";
    }

    backups_map::iterator prepare_output(const string& resource_name)
    {
        string filename = get_filename(resource_name);
        auto ibackup = backups_.find(resource_name);
        if (ibackup != backups_.end() && boost::filesystem::exists(filename))
        {
            return ibackup;
        }

        auto backup_file = boost::make_shared<std::ofstream>();
        backup_file->open(filename);

        if (not backup_file->is_open())
        {
            YLOG_G(error) << "[sync_file] error=\"can't open backup-file\" filename=\"" << filename
                          << "\"";
            return backups_.end();
        }

        // replace existing value if exists
        std::pair<backups_map::iterator, bool> insert_res =
            backups_.insert(std::make_pair(resource_name, backup_file));
        ibackup = insert_res.first;
        if (!insert_res.second)
        {
            ibackup->second = backup_file;
        }
        return ibackup;
    }

private:
    const string basepath_;
    const PositionType default_;
    backups_map backups_;
    mutex mutex_;
};

}}