#pragma once

#include <equalizer/context.h>
#include <equalizer/operation.h>
#include <equalizer/notifications_sender.h>
#include <processor/pipeline_types.h>
#include <ymod_httpclient/client.h>

namespace yxiva { namespace equalizer {

using namespace pipeline;
using notifications_sender = notifications_sender_template<yhttp::client>;

struct equalizer_settings
{
    notifications_sender::settings sender;
    StreamSettings stream;

    std::size_t max_finished_parts() const
    {
        return stream.window - 1;
    }
};

class equalizer
    : public Processor<StreamStrand<operation_ptr>>
    , public yplatform::log::contains_logger
{
    typedef Processor<StreamStrand<operation_ptr>> base_t;
    typedef typename base_t::stream_t stream_t;
    typedef typename stream_t::strand_ptr strand_ptr;

public:
    equalizer(
        boost::asio::io_service& io,
        const equalizer_settings& init_settings,
        const string& db_name,
        const std::shared_ptr<yhttp::client>& http_client,
        const yplatform::log::source& logger = yplatform::log::source())
        : base_t(io, init_settings.stream)
        , yplatform::log::contains_logger(logger)
        , sender_(std::make_shared<notifications_sender>(
              io,
              init_settings.sender,
              db_name,
              http_client,
              logger))
    {
        input()->label("equalizer");
    }

    void start()
    {
        base_t::start();
    }

    void stop()
    {
        sender_->stop();
        base_t::stop();
        YLOG_G(info) << "[equalizer] stopped";
    }

    string name() const
    {
        return "equalizer";
    }

protected:
    void on_data(stream_ptr stream, std::size_t begin_id, std::size_t end_id) noexcept
    {
        for (std::size_t id = begin_id; id < end_id; id++)
        {
            operation_ptr op = stream->at(id);
            assert(op->stream_id == id);

            if (op->action_type == action_t::TRACE_PIPELINE)
            {
                commit(op, stream);
                continue;
            }
            send_notifications(op);
        }
    }

private:
    void send_notifications(operation_ptr op)
    {
        sender_->send_notification(
            op, boost::bind(&equalizer::commit, get_shared_from_this(), op, nullptr));
    }

    void commit(operation_ptr op, stream_ptr stream = nullptr)
    {
        op->ctx->profiler().pop();
        if (op->action_type != action_t::TRACE_PIPELINE)
        {
            YLOG_CTX_LOCAL(op->ctx, info)
                << "operation processed: " << op->ctx->profiler().get_values_with_total() << " lag_"
                << sender_->db_name() << "=" << op->lag() << ".000";
        }

        if (!stream) stream = input();
        if (stream) stream->commit(op->stream_id);
    }

    std::shared_ptr<equalizer> get_shared_from_this()
    {
        return std::dynamic_pointer_cast<equalizer>(this->shared_from_this());
    }

private:
    notifications_sender::ptr sender_;
    strand_ptr strand_;
};

}}
