#pragma once

#include <map>

#include <yplatform/log.h>

namespace yxiva {

#define POSTCALLER_L(severity) YLOG_G(severity) << "[post_caller] "

class post_caller
    : public yplatform::log::contains_logger
    , public std::enable_shared_from_this<post_caller>
{
    typedef time_traits::timer timer_t;
    typedef std::shared_ptr<timer_t> timer_ptr;
    typedef std::map<string, timer_ptr> timers_collection_t;

public:
    post_caller(boost::asio::io_service& io, const time_duration& repeat_interval)
        : io_(io), repeat_interval_(repeat_interval), stopped_(false)
    {
    }

    void post(const string& id, const boost::function<void(void)>& function)
    {
        scoped_lock lock(mutex_);
        if (stopped_) return;

        auto itimer = active_timers_.find(id);
        if (itimer == active_timers_.end())
        {
            timer_ptr timer = std::make_shared<timer_t>(io_);
            active_timers_.insert(itimer, std::make_pair(id, timer));
            timer->expires_from_now(repeat_interval_);
            timer->async_wait(
                boost::bind(&post_caller::on_timer, shared_from_this(), _1, id, function));
            lock.unlock();
        }
        else
        {
            lock.unlock();
            POSTCALLER_L(error) << "attempt to retry duplicate id " << id;
        }
    }

    void on_timer(
        const boost::system::error_code& err,
        const string& id,
        const boost::function<void(void)>& function)
    {
        scoped_lock lock(mutex_);
        if (stopped_) return;

        active_timers_.erase(id);
        if (err == boost::asio::error::operation_aborted) return;
        lock.unlock();

        try
        {
            function();
        }
        catch (const std::exception& ex)
        {
            POSTCALLER_L(error) << "error in async function: id=" << id << " exception=\""
                                << ex.what() << "\"";
        }
        catch (...)
        {
            POSTCALLER_L(error) << "error in async function: id=" << id << " exception=\"unknown\"";
        }
    }

    void stop()
    {
        scoped_lock lock(mutex_);
        stopped_ = true;
        timers_collection_t old_timers;
        active_timers_.swap(old_timers);
        lock.unlock();

        for (auto iter = old_timers.begin(); iter != old_timers.end(); iter++)
        {
            POSTCALLER_L(debug) << "cancel retry: id=" << iter->first;
            iter->second->cancel();
        }
    }

private:
    boost::asio::io_service& io_;
    time_duration repeat_interval_;

    mutex mutex_;
    timers_collection_t active_timers_;
    bool stopped_;
};

}