#pragma once

#include <yxiva/core/authorizer.h>
#include <yxiva/core/auth/error.h>
#include <yxiva/core/auth/caching_authorizer.h>

#include <equalizer/context.h>
#include <equalizer/operation.h>
#include <processor/pipeline_types.h>
#include <processor/post_caller.h>

namespace yxiva { namespace equalizer {

using namespace pipeline;

class uid_fetcher
    : public Processor<StreamStrand<operation_ptr>>
    , public yplatform::log::contains_logger
{
    typedef Processor<StreamStrand<operation_ptr>> base_t;
    typedef uid_fetcher this_t;
    typedef base_t::stream_t stream_t;
    typedef stream_t::strand_ptr strand_ptr;

public:
    struct settings
    {
        string my_host_ip;
        time_duration retry_interval = seconds(2);
        std::size_t cache_size; // used in impl.cc only for now
        StreamSettings stream;
    };

    uid_fetcher(
        boost::asio::io_service& io,
        const settings& settings,
        authorizer_ptr autorizer,
        const yplatform::log::source& logger = yplatform::log::source())
        : base_t(io, settings.stream)
        , yplatform::log::contains_logger(logger)
        , settings_(settings)
        , authorizer_(autorizer)
        , strand_(io)
    {
        retrier_ = std::make_shared<post_caller>(io, settings_.retry_interval);
        input()->label("uid_fetcher_stream");
    }

    void stop()
    {
        retrier_->stop();
        base_t::stop();
        YLOG_L(info) << "[uid_fetcher] stopped";
    }

    string name() const
    {
        return "uid_fetcher";
    }

protected:
    void on_data(stream_ptr stream, std::size_t begin_id, std::size_t end_id) noexcept
    {
        static const std::set<action_t> skip_actions = { action_t::UNKNOWN,
                                                         action_t::TRACE_PIPELINE };
        for (std::size_t id = begin_id; id < end_id; id++)
        {
            auto operation = stream->at(id);
            assert(operation->stream_id == id);

            if ((skip_actions.count(operation->action_type) == 0) &&
                (operation->uid().empty() ^ operation->suid().empty()))
            {
                strand_.post(
                    boost::bind(&this_t::enqueue_auth_request, get_shared_from_this(), operation));
                continue;
            }
            commit_operation(stream, operation);
        }
    }

    void commit_operation(stream_ptr stream, operation_ptr operation)
    {
        operation->ctx->profiler().extrude_and_push("auth_output");
        if (stream) stream->commit(operation->stream_id);
    }

    void commit_operation(operation_ptr operation)
    {
        commit_operation(input(), operation);
    }

    void enqueue_auth_request(operation_ptr operation)
    {
        assert(strand_.running_in_this_thread());
        operation->ctx->profiler().extrude_and_push("auth_request");
        const string id = get_request_id(operation->ui);
        auto irequests = active_requests_.find(id);
        if (irequests != active_requests_.end())
        {
            irequests->second.push_back(operation);
        }
        else
        {
            active_requests_[id].push_back(operation);
            request_auth(operation->ui, operation->ctx);
        }
    }

    void request_auth(const user_info& ui, context_ptr ctx)
    {
        user_info_future_t fres;
        if (static_cast<string>(ui.uid).empty())
        {
            fres = authorizer_->authenticate(ctx, { settings_.my_host_ip, 0 }, ui.suid);
        }
        else
        {
            fres = authorizer_->authenticate(ctx, { settings_.my_host_ip, 0 }, ui.uid);
        }

        fres.add_callback(boost::bind(&this_t::handle_auth, get_shared_from_this(), fres, ui, ctx));
    }

    void fill_operations(const string& request_id, const user_info& ui) noexcept
    {
        assert(strand_.running_in_this_thread());
        auto irequests = active_requests_.find(request_id);
        if (irequests != active_requests_.end())
        {
            for (auto& operation : irequests->second)
            {
                operation->ui = ui;
                commit_operation(operation);
            }
            active_requests_.erase(irequests);
        }
    }

    void skip_operations(const string& request_id) noexcept
    {
        assert(strand_.running_in_this_thread());
        auto irequests = active_requests_.find(request_id);
        if (irequests != active_requests_.end())
        {
            for (auto& operation : irequests->second)
            {
                commit_operation(operation);
            }
            active_requests_.erase(irequests);
        }
    }

    void handle_auth(user_info_future_t fres, const user_info& ui, context_ptr ctx) noexcept
    {
        try
        {
            user_info resp_ui = fres.get();
            strand_.post(boost::bind(
                &this_t::fill_operations, get_shared_from_this(), get_request_id(ui), resp_ui));
            return;
        }
        catch (const no_such_user& e)
        {
            strand_.post(
                boost::bind(&this_t::skip_operations, get_shared_from_this(), get_request_id(ui)));
            return;
        }
        catch (const std::exception& e)
        {
            YLOG_CTX_LOCAL(ctx, error) << "error while auth: exception=\"" << e.what() << "\"";
        }
        catch (...)
        {
            YLOG_CTX_LOCAL(ctx, error) << "error while auth: exception=\"unknown\"";
        }

        retrier_->post(
            get_request_id(ui),
            boost::bind(&this_t::request_auth, get_shared_from_this(), ui, ctx));
    }

    static const string get_request_id(const user_info& ui)
    {
        if (static_cast<string>(ui.uid).empty()) return "s:" + static_cast<string>(ui.suid);
        return "u:" + static_cast<string>(ui.uid);
    }

private:
    std::shared_ptr<uid_fetcher> get_shared_from_this()
    {
        return std::dynamic_pointer_cast<uid_fetcher>(this->shared_from_this());
    }

private:
    settings settings_;
    authorizer_ptr authorizer_;
    std::shared_ptr<post_caller> retrier_;

    boost::asio::io_service::strand strand_;
    std::map<string, std::vector<operation_ptr>> active_requests_;
};

}}
