#include "decode_json.h"

#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/numeric/conversion/cast.hpp>

#include <yplatform/log.h>

#include <generator/pg_action_type.h>

namespace yxiva { namespace equalizer { namespace decoder {

namespace {

string json_value_to_string(const json_value& value)
{
    return value.stringify();
}

} // namespace

void decode_json(equalizer::operation& operation, const string& data)
{
    static const std::set<string> expected_array_fields = { "message_flags", "operation_flags",
                                                            "message_types", "operation_types",
                                                            "lids",          "to",
                                                            "reply-to",      "cc",
                                                            "bcc",           "reply-to-all" };

    operation.parts.push_back(json_parse(data));
    json_value& body = operation.parts.front();
    for (auto item = body.members_begin(); item != body.members_end(); item++)
    {
        string key(item.key());
        boost::algorithm::to_lower(key);
        auto&& value = *item;

        string str_value = json_value_to_string(value);
        if (key == "uname")
        {
            boost::trim(str_value);
            operation.ui.suid = str_value;
        }
        else if (key == "lcn")
        {
            boost::trim(str_value);
            operation.lcn = str_value;
        }
        else if (key == "operation_id")
        {
            operation.operation_id = cast_json_value<std::size_t>(value, key);
        }
        else if (key == "action_type")
        {
            operation.action_type =
                resolve_action_type(json_get<string>(value, key.data(), string{}), operation.args);
        }
        else if (key == "operation_size")
        {
            operation.total_count = cast_json_value<std::size_t>(value, key);
        }
        else if (key == "operation_timestamp")
        {
            operation.ts = cast_json_value<std::time_t>(value, key);
        }
        else if (key == "uid")
        {
            boost::trim(str_value);
            operation.ui.uid = str_value;
        }
        else if (key == "session_key")
        {
            operation.args[key] = str_value;
        }
    }

    for (auto key = expected_array_fields.begin(); key != expected_array_fields.end(); key++)
    {
        if (!body.has_member(*key) || body[*key].empty())
        {
            body[*key] = "[]";
        }
    }
}

}}}
