#pragma once

#include <map>
#include <yxiva/core/json.h>
#include <boost/numeric/conversion/cast.hpp>

#include <equalizer/operation.h>

namespace yxiva { namespace equalizer { namespace decoder {

using std::map;

template <typename T>
T uint_cast(const string& source, const string& name)
{
    try
    {
        long long value = boost::lexical_cast<long long>(source);
        return boost::numeric_cast<T>(value);
    }
    catch (const std::bad_cast& ex)
    {
        throw std::runtime_error(string("can't cast ") + name);
    }
}

template <typename T>
T positive_int_cast(long long source, const string& name)
{
    if (source <= 0) throw std::runtime_error(string(name) + " must be > 0");
    try
    {
        return boost::numeric_cast<T>(source);
    }
    catch (const std::bad_cast& ex)
    {
        throw std::runtime_error(string("can't cast ") + name);
    }
}

template <typename T>
T positive_int_cast(const string& source, const string& name)
{
    try
    {
        long long value = boost::lexical_cast<long long>(source);
        return positive_int_cast<T>(value, name);
    }
    catch (const std::bad_cast& ex)
    {
        throw std::runtime_error(string("can't cast ") + name);
    }
}

template <typename T>
T cast_json_value(const json_value& value, const string& name)
{
    if (value.is_string()) return positive_int_cast<T>(value.to_string(), name);

    if (value.is_int64()) return positive_int_cast<T>(value.to_int64(), name);

    if (value.is_uint64()) return positive_int_cast<T>(value.to_uint64(), name);

    throw std::runtime_error(string("can't cast ") + name);
}

void decode_json(equalizer::operation& operation, const string& data);

}}}
