#include "notify.h"

#include <sstream>
#include <boost/bind/protect.hpp>

#include <ymod_webserver/codes.h>
#include <yxiva/core/authorizer.h>
#include <yxiva/core/auth/error.h>

#include <equalizer/operation.h>
#include <equalizer/equalizer_result.h>
#include <web/decoder/decode_json.h>

namespace yxiva { namespace equalizer { namespace web { namespace methods {

namespace {

void make_http_response(
    const boost::system::error_code& err,
    operation_state st,
    ymod_webserver::request_ptr req,
    ymod_webserver::response_ptr resp)
{
    if (err)
    {
        YLOG_CTX_GLOBAL(req->ctx(), error) << "error while processing: " << err.message();
        resp->set_code(ymod_webserver::codes::bad_request);
        return;
    }
    else
    {
        switch (st)
        {
        case operation_state::finished:
            resp->set_code(ymod_webserver::codes::ok);
            break;
        case operation_state::seen:
        case operation_state::ignored:
            resp->result(ymod_webserver::codes::accepted);
            break;
        default:
            resp->set_code(ymod_webserver::codes::bad_gateway);
            break;
        }
    }
}

void handle_auth(
    user_info_future_t fres,
    ymod_webserver::request_ptr req,
    ymod_webserver::response_ptr resp,
    equalizer::operation& operation,
    const string& sequence_id,
    settings_ptr settings)
{
    try
    {
        operation.ui = fres.get();
    }
    catch (const no_such_user& e)
    {
        YLOG_CTX_GLOBAL(req->ctx(), info) << "handle_auth error: " << e.what();
        // resp->result(ymod_webserver::codes::bad_request, "no such user");
    }
    catch (const authorizer_error& e)
    {
        YLOG_CTX_GLOBAL(req->ctx(), info) << "yxiva::error: " << e.what();
        resp->result(ymod_webserver::codes::bad_gateway, "authentication service not available");
        return;
    }
    catch (const std::exception& e)
    {
        YLOG_CTX_GLOBAL(req->ctx(), error) << "unexpected error while auth: " << e.what();
        resp->result(ymod_webserver::codes::bad_gateway, "authentication failed");
        return;
    }
    catch (...)
    {
        YLOG_CTX_GLOBAL(req->ctx(), error) << "unexpected error while auth: unknown";
        resp->result(ymod_webserver::codes::bad_gateway, "authentication failed");
        return;
    }
    settings->processor_module->manualy_put_operation(
        sequence_id, operation, boost::bind(make_http_response, _1, _2, req, resp));
}

}

#define THROW_IF(condition, field)                                                                 \
    if (condition)                                                                                 \
    {                                                                                              \
        throw std::runtime_error(string(field) + " not specified");                                \
    }

void notify(
    ymod_webserver::request_ptr req,
    ymod_webserver::response_ptr resp,
    settings_ptr settings)
{
    equalizer::operation operation(req->ctx()->uniq_id());
    string data(req->raw_body.begin(), req->raw_body.end());

    try
    {
        decoder::decode_json(operation, data);

        string id_str = req->headers["zooqueueid"];
        if (id_str.empty()) id_str = req->url.param_value("zooqueueid", "");
        operation.stream_id = decoder::uint_cast<std::size_t>(id_str, "ZooQueueId");

        THROW_IF(
            operation.suid().empty() and operation.uid().empty(),
            "must be set at least one of: uname, uid");
        THROW_IF(operation.lcn.empty(), "lcn");
        THROW_IF(not operation.operation_id, "operation_id");
        THROW_IF(not operation.action_type, "action_type");
        THROW_IF(not operation.total_count, "operation_size");
    }
    catch (const std::exception& ex)
    {
        YLOG_CTX_GLOBAL(req->ctx(), error) << "notify error: bad_request " << ex.what();
        resp->result(ymod_webserver::codes::bad_request, ex.what());
        return;
    }

    string sequence_id = req->headers["zooshardid"];
    if (sequence_id.empty()) sequence_id = req->url.param_value("zooshardid", "");
    if (sequence_id.empty())
    {
        YLOG_CTX_GLOBAL(req->ctx(), error) << "empty header: ZooShardId";
        resp->result(ymod_webserver::codes::bad_request, "empty header: ZooShardId");
        return;
    }

    YLOG_CTX_GLOBAL(req->ctx(), info)
        << "operation: sequence_id=\"" << sequence_id
        << "\""
           " stream_id="
        << operation.stream_id << " operation_id=" << operation.operation_id << " uid=\""
        << operation.uid()
        << "\""
           " suid=\""
        << operation.suid()
        << "\""
           " lcn=\""
        << operation.lcn
        << "\""
           " action=\""
        << operation.action_type
        << "\""
           " size="
        << operation.total_count;

    YLOG_CTX_GLOBAL(req->ctx(), debug) << "operation body=\"" << data << "\"";

    if (operation.uid().empty())
    {
        user_info_future_t fres = settings->authorizer->authenticate(
            req->ctx(),
            { req->context->remote_address, req->context->remote_port },
            operation.ui.suid);
        fres.add_callback(
            boost::bind(handle_auth, fres, req, resp, operation, sequence_id, settings));
    }
    else if (operation.suid().empty())
    {
        user_info_future_t fres = settings->authorizer->authenticate(
            req->ctx(),
            { req->context->remote_address, req->context->remote_port },
            operation.ui.uid);
        fres.add_callback(
            boost::bind(handle_auth, fres, req, resp, operation, sequence_id, settings));
    }
    else
    {
        settings->processor_module->manualy_put_operation(
            sequence_id, operation, boost::bind(make_http_response, _1, _2, req, resp));
    }
}

}}}}
