#!/usr/bin/env python3

import json
import re
import subprocess
import time

def print_metrics_from_statport():
    stats = read_stats(8091)
    modules = stats['stat']['modules']
    ts = int(time.time())
    for name, stats in modules.items():
        if name.startswith('pq_') and len(stats) > 0:
            print_pq_metrics(name, stats, ts)

def read_stats(stat_port):
    json_text = subprocess.check_output(['nc', 'localhost', str(stat_port)],
        universal_newlines=True)
    return json.loads(json_text)

def is_cluster_stats(stats):
    return 'master' in stats

def print_pq_metrics(module_name, stats, ts):
    for pool_name, pool_stats in stats.items():
        if is_cluster_stats(pool_stats):
            for host, host_stats in pool_stats.items():
                if host != 'monitor':
                    if host not in ['default_replica', 'master']:
                        host = host[:3]
                    role = 'master' if host_stats['is_master'] == 'true' else 'replica.%s' % host
                    dbname = extract_dbname(host_stats['conninfo'])
                    print_health_metrics(module_name, host_stats, role, ts, dbname)
                    print_single_pool_metrics(module_name, host_stats['pq']['pool_0'], role, ts, dbname)
            print_lag_metrics(module_name, pool_stats['monitor'], ts, dbname)
        else:
            dbname = extract_dbname(pool_stats['conninfo'])
            print_single_pool_metrics(module_name, pool_stats, 'master.master', ts, dbname)

def print_lag_metrics(module_name, stats, ts, dbname):
    replicas = stats['replicas']
    # This happens if replica monitor is not working for some reason.
    if len(replicas) == 0:
        return
    for replica, lag in replicas.items():
        print_metric('db.timings.' + module_name + '.replica.' + replica[:3] + '.lag', [float(lag)], ts, dbname)

def print_health_metrics(module_name, host_stats, host, ts, dbname):
    print_metric('db.' + module_name + '.' + host + '.state.' + host_stats['state'], 1, ts, dbname)
    print_metric('db.' + module_name + '.' + host + '.error_rate', host_stats['error_rate'], ts, dbname)
    print_metric('db.' + module_name + '.' + host + '.rps', host_stats['rps'], ts, dbname)

def print_single_pool_metrics(module_name, pool_stats, host, ts, dbname):
    print_metric('db.' + module_name + '.' + host + '.connections.busy', pool_stats['busy'], ts, dbname)
    print_metric('db.' + module_name + '.' + host + '.connections.idle', pool_stats['idle'], ts, dbname)
    print_metric('db.' + module_name + '.' + host + '.connections.pending', pool_stats['pending'], ts, dbname)
    print_metric('db.' + module_name + '.' + host + '.queue', pool_stats['queue_size'], ts, dbname)

def extract_dbname(conninfo):
    m = re.search(r'dbname=([a-zA-Z0-9_]+)', conninfo)
    return m.group(1)

def print_metric(name, value, ts, tier):
    print('%s %s %s %s' % (name, value, ts, (tier if tier is not None else '')))

try:
    print_metrics_from_statport()
except:
    # Nothing to do, silently lose metrics.
    pass