#pragma once

#include <yxiva/core/split.h>
#include <yxiva/core/batch_key.h>
#include <boost/functional/hash.hpp>

#include <ymod_xstore/interface.h>

namespace yxiva { namespace hub { namespace api {
namespace hacks {

inline bool force_master(const std::string& service)
{
    // TODO: find a better solution, ticket: RTEC-3556.
    static const std::unordered_set<std::string> master_requiring_services = {
        "messenger",           "messenger-prod",      "telemost-test",
        "telemost-test2",      "telemost-mediator",   "telemost-mediator-2",
        "telemost-mediator-3", "telemost-mediator-4", "telemost-mediator-yt"
    };
    return master_requiring_services.count(service) > 0;
}

}

using ymod_webserver::http::expirable_stream_ptr;

struct notify_status
{
    size_t seq = 0;
    int code = -1;
    string body;
    string subscription_id;

    MSGPACK_DEFINE(seq, code, body, subscription_id);
};

class root_notify_context
{
    // Key is (uid, id) pair. Reference is used to prevent string copying
    // on lookup operations when creating key object. Actual strings are stored separately.
    using key_type = std::pair<const string&, const string&>;

public:
    root_notify_context(expirable_stream_ptr stream, const batch_keys& k)
        : stream(stream)
        , keys(k)
        , subscription_counters(keys.size(), 0U)
        , timer(stream->io_service())
        , strand(stream->io_service())
    {
        for (size_t i = 0; i < keys.size(); ++i)
        {
            keys_inverted_index[key_type{ keys[i].uid, keys[i].subscription_id }] = i;
        }
    }

    boost::optional<size_t> try_get_index(const string& uid, const string& subscription_id)
    {
        auto it = keys_inverted_index.find(key_type{ uid, subscription_id });
        if (it == keys_inverted_index.end())
        {
            it = keys_inverted_index.find(key_type{ uid, string() });
        }
        if (it == keys_inverted_index.end())
        {
            return boost::none;
        }
        return it->second;
    }

    size_t get_index(const string& uid, const string& subscription_id)
    {
        auto it = keys_inverted_index.find(key_type{ uid, subscription_id });
        if (it != keys_inverted_index.end()) return it->second;

        it = keys_inverted_index.find(key_type{ uid, string() });
        if (it != keys_inverted_index.end()) return it->second;

        throw std::runtime_error("no element (" + uid + ", " + subscription_id + ")");
    }

    expirable_stream_ptr stream;
    batch_keys keys;
    // For lookup by (uid, id) pair.
    std::unordered_map<key_type, size_t, boost::hash<key_type>> keys_inverted_index;
    std::vector<unsigned> subscription_counters;
    time_traits::timer timer;
    boost::asio::io_service::strand strand;
    size_t total_results = 0;
    bool released = false;
    std::vector<notify_status> results;
};

// Calls convey in parallel if multiple xtable::find results.
struct batch_notify_coro : public boost::asio::coroutine
{
    std::shared_ptr<state> hub;
    expirable_stream_ptr stream;
    shared_ptr<::yxiva::message> message;
    batch_keys keys;                         // TODO refactor
    time_duration send_result_timeout_delta; // to send response before ctx.deadline
    boost::shared_ptr<root_notify_context> root_ctx;

    auto find_options()
    {
        XTable::find_options options;
        options.db_role = hacks::force_master(message->service) ? XTable::db_role::master :
                                                                  XTable::db_role::replica;
        return options;
    }

    void operator()(const error_code& err, const batch_iterators& batch, sub_list list)
    {
        (*this)(err, &batch, &list);
    }

    void operator()(const error_code& err, sub_list list)
    {
        batch_iterators batch = { root_ctx->keys.begin() };
        (*this)(err, &batch, &list);
    }

    void operator()(
        const error_code& err = error_code(),
        const batch_iterators* batch = nullptr,
        sub_list* subscriptions = nullptr)
    {
        // root_ctx may be not initialized at start.
        if (root_ctx && root_ctx->keys.empty())
        {
            root_ctx->timer.cancel();
            return;
        }

        try
        {
            reenter(this)
            {

                root_ctx = boost::make_shared<root_notify_context>(stream, std::move(keys));
                keys.clear(); // to not copy keys on every async op, TODO refactor
                root_ctx->timer.expires_at(stream->ctx()->deadline() - send_result_timeout_delta);
                root_ctx->timer.async_wait(
                    root_ctx->strand.wrap(boost::bind(&batch_notify_coro::send_results, *this)));

                root_ctx->total_results += root_ctx->keys.size();

                if (root_ctx->keys.size() > 1)
                {
                    yield hub->xtable->batch_find(
                        stream->ctx(),
                        root_ctx->keys,
                        message->service,
                        find_options(),
                        root_ctx->strand.wrap(*this));
                }
                else
                {
                    yield hub->xtable->find(
                        stream->ctx(),
                        root_ctx->keys[0].uid,
                        message->service,
                        find_options(),
                        root_ctx->strand.wrap(*this));
                    if (!err && root_ctx->keys[0].subscription_id.size())
                    {
                        filter_by_subscription_id(
                            *subscriptions, root_ctx->keys[0].subscription_id);
                    }
                }

                if (err)
                {
                    add_batch_results(err, *batch);
                    return;
                }

                if (subscriptions->empty())
                {
                    add_batch_results(make_error_code(err_code_no_subscriptions), *batch);
                    return;
                }

                root_ctx->total_results -= batch->size();
                root_ctx->total_results += subscriptions->size();

                auto missing_keys = find_missing_keys(*subscriptions, *batch);
                if (missing_keys.size())
                {
                    root_ctx->total_results += missing_keys.size();
                    add_batch_results(make_error_code(err_code_no_subscriptions), missing_keys);
                }

                // Handle application stop.
                if (stream->ctx()->is_cancelled())
                {
                    root_ctx->timer.cancel();
                    return;
                }

                auto convey_ctx =
                    boost::make_shared<convey_context>(stream->ctx(), message, *subscriptions);

                convey_ctx->deadline(root_ctx->timer.expires_at());

                // No yield.
                hub->exhaust->batch_convey(convey_ctx, root_ctx->strand.wrap(*this));
            }
        }
        catch (const std::exception& e)
        {
            YLOG_CTX_GLOBAL(stream->ctx(), error) << "batch_notify_coro exception: " << e.what();
        }
    }

    void filter_by_subscription_id(sub_list& subscriptions, string subscription_id)
    {
        subscriptions.erase(
            std::remove_if(
                subscriptions.begin(),
                subscriptions.end(),
                [&subscription_id](const sub_t& sub) -> bool { return sub.id != subscription_id; }),
            subscriptions.end());
    }

    batch_iterators find_missing_keys(sub_list& subscriptions, const batch_iterators& batch)
    {
        for (const auto& sub : subscriptions)
        {
            auto index = root_ctx->try_get_index(sub.uid, sub.id);
            if (!index) continue;
            ++root_ctx->subscription_counters[index.get()];
        }

        batch_iterators missing; // TODO reserve batch.size() - keys_count?
        for (const auto& i : batch)
        {
            if (root_ctx->subscription_counters[i - root_ctx->keys.begin()] == 0)
            {
                missing.push_back(i);
            }
        }
        return missing;
    }

    // Executed in strand.
    void operator()(const error_code& code, const sub_t& subscription, const string& body)
    {
        if (root_ctx->released) return;
        notify_status status;
        status.seq = root_ctx->get_index(subscription.uid, subscription.id);
        status.code = http_code_for_error(code);
        status.body = body.size() ? body : (code ? code.message() : "OK");
        status.subscription_id = subscription.id;
        root_ctx->results.push_back(std::move(status));
        if (root_ctx->results.size() >= root_ctx->total_results)
        {
            root_ctx->timer.cancel(); // Will send results.
        }
    }

    // Executed in strand.
    void add_batch_results(const error_code& code, const batch_iterators& batch)
    {
        if (root_ctx->released) return;

        root_ctx->results.reserve(root_ctx->results.size() + batch.size());
        notify_status status;
        for (auto it : batch)
        {
            status.seq = root_ctx->get_index(it->uid, it->subscription_id);
            status.code = http_code_for_error(code);
            status.body = status.code == 500 ? "internal error" : code.message();
            status.subscription_id = it->subscription_id;
            root_ctx->results.push_back(std::move(status));
        }

        if (root_ctx->results.size() >= root_ctx->total_results)
        {
            root_ctx->timer.cancel(); // Will send results.
        }
    }

    // Executed in strand. Ignores asio timer error code (sends anyway).
    void send_results()
    {
        if (root_ctx->released) return;
        root_ctx->released = true;

        auto&& ctx = root_ctx->stream->ctx();
        ctx->cancel();

        if (root_ctx->results.size() < root_ctx->total_results)
        {
            YLOG_CTX_GLOBAL(ctx, info) << "send incomplete result";
        }
        else
        {
            YLOG_CTX_GLOBAL(ctx, info) << "send result";
        }

        if (auto http_stream = stream->detach_stream())
        {
            http_stream->set_code(ymod_webserver::codes::ok);
            http_stream->add_header("NotificationID", std::to_string(message->local_id));
            http_stream->add_header("TransitID", message->transit_id);
            http_stream->result_body(pack(root_ctx->results));
        }
    }
};

}}}
