#include "bind.h"

#include "api/add_broken_subscription.h"
#include "api/batch_notify.h"
#include "api/control.h"
#include "api/convey_check.h"
#include "api/get_counters.h"
#include "api/get_history.h"
#include "api/is_master.h"
#include "api/list.h"
#include "api/list_json.h"
#include "api/notify.h"
#include "api/ping.h"
#include "api/repeat_messages.h"
#include "api/status.h"
#include "api/send.h"
#include "api/degrade_control.h"
#include "api/subscribe.h"
#include "api/topology.h"
#include "api/unsubscribe.h"
#include "api/unsubscribe_overlapped.h"
#include "api/update_callback.h"
#include "api/shards.h"
#include "api/update_topology.h"
#include "api/xtasks.h"
#include "api/resharding/control.h"
#include "api/resharding/execute_xtable.h"

#include <ymod_webserver/server.h>

namespace yxiva { namespace hub {

namespace {

using stream_ptr = ymod_webserver::http::stream_ptr;
using expirable_stream_ptr = ymod_webserver::http::expirable_stream_ptr;

inline stream_ptr identity_transform(const stream_ptr& stream)
{
    return stream;
}

inline std::function<expirable_stream_ptr(stream_ptr)> make_expirable_stream_transform(
    time_traits::duration timeout)
{
    return [base_timeout = timeout](stream_ptr stream) {
        auto timeout =
            std::min(base_timeout, stream->ctx()->deadline() - time_traits::clock::now());
        timeout = std::max(timeout, time_traits::duration::zero());
        return ymod_webserver::http::make_expirable_stream(stream, timeout);
    };
}

// Helper for api method call. Creates a new coro or functor instance
// and calls with all passed arguments.
template <typename T>
struct executer
{
    template <typename... Args>
    void operator()(Args&&... args)
    {
        if constexpr (std::is_base_of_v<boost::asio::coroutine, T>)
        {
            T t = { {}, std::forward<Args>(args)... };
            t();
        }
        else
        {
            T t = { std::forward<Args>(args)... };
            t();
        }
    }
};

// TODO extract and rename
template <typename Func, typename StreamTransform>
struct wrapper
{
    // Takes a functor only for caller's code readabylity, see bind_api().
    wrapper(Func /*f*/, StreamTransform sf, std::shared_ptr<state> state)
        : stream_transform_(sf), state_(state)
    {
    }

    template <typename Stream, typename... CallArgs>
    void operator()(Stream&& s, CallArgs&&... callargs)
    {
        executer<std::decay_t<Func>>{}(state_, stream_transform_(s), callargs...);
    }

    StreamTransform stream_transform_;
    std::shared_ptr<state> state_;
};

struct no_auth
{
    template <typename Handler>
    auto operator()(settings_ptr /*settings*/, Handler&& h)
    {
        return std::forward<Handler>(h);
    }
};

struct tvm_auth
{
    template <typename Handler>
    auto operator()(settings_ptr settings, Handler&& h)
    {
        return check_tvm(settings->tvm, std::forward<Handler>(h));
    }
};

template <typename State>
class bind_helper
{
public:
    bind_helper(State shared_state) : shared_state_(shared_state)
    {
    }

    template <typename F, typename... Args>
    inline void plain(const std::vector<string>& path, F f, Args... args)
    {
        bind_impl(path, f, identity_transform, no_auth{}, args...);
    }

    template <typename F, typename... Args>
    inline void plain_tvm(const std::vector<string>& path, F f, Args... args)
    {
        bind_impl(path, f, identity_transform, tvm_auth{}, args...);
    }

    template <typename F, typename... Args>
    inline void expirable(
        const std::vector<string>& path,
        F f,
        time_traits::duration timeout,
        Args... args)
    {
        bind_impl(path, f, make_expirable_stream_transform(timeout), no_auth{}, args...);
    }

    template <typename F, typename... Args>
    inline void expirable_tvm(
        const std::vector<string>& path,
        F f,
        time_traits::duration timeout,
        Args... args)
    {
        bind_impl(path, f, make_expirable_stream_transform(timeout), tvm_auth{}, args...);
    }

private:
    template <typename F, typename StreamTransform, typename Auth, typename... Args>
    inline void bind_impl(
        const std::vector<string>& path,
        F f,
        StreamTransform st,
        Auth auth,
        Args... args)
    {
        auto webserver_module = yplatform::find<ymod_webserver::server>("web_server");
        webserver_module->bind(
            "front", path, auth(shared_state_->settings, wrapper(f, st, shared_state_)), args...);
    }

    State shared_state_;
};

inline bool starts_with(const string& str, const string& prefix)
{
    return prefix.size() <= str.size() && std::equal(prefix.begin(), prefix.end(), str.begin());
}

// TODO extract to webserver
struct splitter
{
    splitter(const string& delimiter = ",") : delimiter(delimiter)
    {
    }

    std::vector<string> operator()(const string& value) const
    {
        std::vector<string> result = utils::split_unique(value, delimiter);
        return result;
    }

    string delimiter;
};

struct db_role_converter
{
    XTable::db_role operator()(const string& source) const
    {
        using namespace std::string_literals;
        using db_role = XTable::db_role;

        if (source == "master"s) return db_role::master;
        if (source == "replica"s) return db_role::replica;
        throw std::runtime_error("unknown db_role " + source);
    }
};

}

void bind_api(std::shared_ptr<state> shared_state)
{
    using ymod_webserver::transformer;
    using ymod_webserver::argument;
    using ymod_webserver::optional_argument;
    using ymod_webserver::default_validator;
    using ymod_webserver::validator;

    auto bind = bind_helper(shared_state);

    auto webserver_module = yplatform::find<ymod_webserver::server>("web_server");
    auto uid_argument = argument<string>("uid");
    auto uid_list_argument = argument<std::vector<string>>(
        "uid", { "uid", "uids" }, default_validator<string>(), splitter());
    auto optional_uid_argument = optional_argument<string>("uid", "");
    auto service_argument = argument<string>("service");
    auto service_list_argument = argument<std::vector<string>>(
        "services", { "service", "services" }, [](const string&) { return true; }, splitter());
    auto db_role_argument = optional_argument<XTable::db_role>(
        "db_role", XTable::db_role::replica, default_validator<string>(), db_role_converter{});
    auto list_transformer =
        transformer(optional_uid_argument, argument<string>("service"), db_role_argument);
    auto batch_list_transformer = transformer(optional_uid_argument, service_list_argument);
    auto batch_uids_list_transformer = transformer(uid_list_argument, service_argument);
    auto xtasks_delay_transformer = transformer(
        argument<string>("worker", validator(isalnum_ext)),
        argument<string>("id"),
        argument<time_duration::rep>("delay_sec"));
    auto notify_transformer = transformer(
        optional_argument<std::vector<string>>("tags", {}, validator(tag_list_char), splitter()),
        optional_argument<std::time_t>("event_ts", 0),
        optional_argument<string>("message_topic", ""),
        optional_argument<bool>("enable_deduplication", false));
    auto resharding_role_arg = argument<resharding::controller::role>(
        "role",
        ymod_webserver::default_validator<string>(),
        ymod_webserver::converter(api::resharding::role_from_string));
    auto xtable_query_type_arg = argument<xtable::resharding::query_type>(
        "type",
        ymod_webserver::default_validator<string>(),
        ymod_webserver::converter(xtable::resharding::query_type_from_string));
    auto request_reaction_arg = argument<resharding::controller::request_reaction>(
        "request_reaction",
        ymod_webserver::default_validator<string>(),
        ymod_webserver::converter(api::resharding::request_reaction_from_string));

    string ep = "front";

    bind.plain({ "/ping" }, api::ping());
    bind.expirable_tvm(
        { "/notify/*" },
        api::simple_notify(),
        shared_state->settings->notify_timeout,
        notify_transformer);
    bind.expirable_tvm(
        { "/fast_notify/*" },
        api::fast_notify(),
        shared_state->settings->notify_timeout,
        notify_transformer);
    bind.expirable_tvm(
        { "/binary_notify", "/binary_notify/*" },
        api::binary_notify(),
        shared_state->settings->notify_timeout,
        notify_transformer);
    bind.expirable_tvm(
        { "/fast_binary_notify", "/fast_binary_notify/*" },
        api::fast_binary_notify(),
        shared_state->settings->notify_timeout,
        notify_transformer);
    bind.expirable_tvm(
        {
            "/batch_binary_notify",
            "/batch_binary_notify_single/*", // size == 1
            "/batch_binary_notify_small/*",
            "/batch_binary_notify_medium/*",
            "/batch_binary_notify/*" // large
        },
        api::batch_binary_notify(),
        shared_state->settings->batch_notify_timeout);
    bind.expirable_tvm({ "/send/*" }, api::send(), shared_state->settings->notify_timeout);
    bind.plain({ "/control" }, api::control_view());

    bind.plain_tvm({ "/list" }, api::list(), list_transformer);
    bind.plain_tvm({ "/list_json" }, api::list_json(), list_transformer);
    bind.plain_tvm({ "/batch_list_json" }, api::batch_list_json(), batch_list_transformer);
    bind.plain_tvm(
        { "/batch_uids_list_json" }, api::batch_uids_list_json(), batch_uids_list_transformer);

    bind.expirable_tvm(
        { "/subscribe", "/uidset/subscribe" },
        api::subscribe(),
        shared_state->settings->subscribe_timeout,
        transformer(
            optional_argument<string>("uid", ""),
            argument<string>("service", validator(isalnum_ext)),
            argument<string>("callback"),
            optional_argument<string>("client", ""),
            optional_argument<string>("session_key", ""),
            optional_argument<string>("filter", ""),
            optional_argument<string>("extra_data", ""),
            optional_argument<ttl_t>("ttl", shared_state->settings->default_ttl),
            optional_argument<string>("id", ""),
            optional_argument<string>("bb_connection_id", ""),
            optional_argument<string>("uidset", ""),
            optional_argument<local_id_t>("position", 0),
            optional_argument<unsigned>("history_count", 0),
            optional_argument<bool>("strict_position", false)));
    bind.expirable(
        { "/subscribe_mobile" },
        api::subscribe_mobile(),
        shared_state->settings->subscribe_timeout,
        transformer(
            optional_argument<string>("uid", ""),
            argument<string>("service", validator(isalnum_ext)),
            argument<string>("callback"),
            optional_argument<string>("client", ""),
            argument<string>("session_key", { "session_key", "uuid" }), // device_uuid
            optional_argument<string>("filter", ""),
            optional_argument<string>("extra_data", ""),
            optional_argument<ttl_t>("ttl", shared_state->settings->default_ttl),
            optional_argument<string>("id", ""),
            optional_argument<local_id_t>("position", 0),
            argument<string>("platform"),
            optional_argument<string>("device", { "device", "device_id" }, ""),
            optional_argument<string>("bb_connection_id", "")));
    bind.plain_tvm(
        { "/repeat_messages" },
        api::repeat_messages(),
        transformer(
            uid_argument,
            service_argument,
            argument<local_id_t>("position"),
            argument<local_id_t>("count")));
    bind.expirable_tvm(
        { "/unsubscribe", "/uidset/unsubscribe" },
        api::unsubscribe(),
        shared_state->settings->unsubscribe_timeout,
        transformer(
            optional_argument<string>("uid", ""),
            argument<string>("service", validator(isalnum_ext)),
            argument<string>(yxiva::parameters::SUBSCRIPTION_ID)));
    bind.expirable_tvm(
        { "/batch_unsubscribe" },
        api::batch_unsubscribe(),
        shared_state->settings->unsubscribe_timeout,
        transformer(
            optional_argument<string>("uid", ""),
            argument<string>("service", validator(isalnum_ext)),
            argument<std::vector<string>>(
                "ids", [](const string&) { return true; }, splitter()),
            optional_argument<std::time_t>("ts", 0)));
    bind.expirable_tvm(
        { "/unsubscribe_mobile" },
        api::unsubscribe_mobile(),
        shared_state->settings->unsubscribe_timeout,
        transformer(
            optional_argument<string>("uid", ""),
            argument<string>("service", validator(isalnum_ext)),
            argument<string>("uuid")));
    bind.plain_tvm(
        { "/unsubscribe_overlapped" },
        api::unsubscribe_overlapped(),
        transformer(
            optional_argument<string>("uid", "", validator(isalnum_ext)),
            argument<string>("service", validator(isalnum_ext)),
            argument<unsigned>("overlap"),
            argument<string>(parameters::SUBSCRIPTION_ID)));
    bind.expirable_tvm(
        { "/add_broken_subscription" },
        api::add_broken_subscription(),
        shared_state->settings->subscribe_timeout,
        transformer(
            argument<string>("platform"),
            argument<string>("id"),
            optional_argument<std::time_t>("timestamp", 0)));
    bind.plain({ "/is_master" }, api::is_master());
    bind.plain({ "/convey_check" }, api::convey_check());
    bind.plain({ "/replica_status", "/state", "/status" }, api::status());
    bind.plain_tvm(
        { "/get_history" },
        api::get_history(),
        transformer(
            argument<string>("uid"),
            service_argument,
            optional_argument<local_id_t>("start_local_id", 0),
            optional_argument<unsigned>("limit", 100000),
            optional_argument<time_t>("begin_ts", 0),
            optional_argument<time_t>("end_ts", 0)));
    bind.plain(
        { "/get_counters" }, api::get_counters(), transformer(uid_argument, service_list_argument));
    bind.plain({ "/enable/*", "/disable/*" }, api::degrade_control());
    bind.plain({ "/topology" }, api::topology());
    bind.plain(
        { "/update_topology" },
        api::update_topology(),
        transformer(argument<string>("peers"), optional_argument<bool>("force_update", false)));
    bind.plain(
        { "/xtasks_create" },
        api::xtasks::create_task(),
        transformer(
            uid_argument,
            service_argument,
            argument<local_id_t>("local_id"),
            optional_argument<bool>("wakeup_on_create", true),
            optional_argument<bool>("ignore_delay_if_pending", true)));
    bind.plain(
        { "/xtasks_start" },
        api::xtasks::get_tasks(),
        transformer(argument<string>("worker"), argument<unsigned>("count")));
    bind.plain({ "/xtasks_summary" }, api::xtasks::get_summary());
    bind.plain(
        { "/xtasks_fin" },
        api::xtasks::fin_task(),
        transformer(argument<string>("worker"), argument<string>("id")));
    bind.plain({ "/xtasks_delay" }, api::xtasks::delay_task(), xtasks_delay_transformer);
    bind.plain(
        { "/xtasks_delay_shallow" }, api::xtasks::delay_task_shallow(), xtasks_delay_transformer);
    bind.plain(
        { "/xtasks_cleanup_active" },
        api::xtasks::cleanup_active(),
        transformer(argument<time_t>("deadline_sec")));
    bind.plain(
        { "/xtasks_cleanup_workers" },
        api::xtasks::cleanup_workers(),
        transformer(argument<time_t>("deadline_sec")));
    bind.plain({ "/xtasks_wakeup_delayed" }, api::xtasks::wakeup_delayed());
    bind.plain({ "/xtasks_alive" }, api::xtasks::alive(), transformer(argument<string>("worker")));
    bind.plain({ "/xtasks_counters" }, api::xtasks::counters());
    bind.plain({ "/xtasks_clear" }, api::xtasks::clear());
    bind.plain_tvm(
        { "/uidset/list" },
        api::list_json_uidset(),
        transformer(argument<string>("uidset"), service_argument));
    bind.expirable_tvm(
        { "/uidset/update_callback" },
        api::update_callback_uidset(),
        shared_state->settings->update_uidset_callback_timeout,
        transformer(argument<string>("uidset"), service_argument, argument<string>("callback")));
    bind.plain(
        { "/xtable/shards", "/xtable/shards_full" },
        api::shards<XTable>(),
        transformer(optional_argument<bool>("human_readable", false)));
    bind.plain(
        { "/xstore/shards", "/xstore/shards_full" },
        api::shards<ymod_xstore::xstore>(),
        transformer(optional_argument<bool>("human_readable", false)));
    bind.plain(
        { "/deduplicate_mobile_subscriptions" },
        api::deduplicate(),
        transformer(uid_argument, service_argument));
    bind.plain_tvm(
        { "/resharding/xtable/execute_query" },
        api::resharding::execute_xtable(),
        transformer(xtable_query_type_arg));
    bind.plain(
        { "/resharding/xtable/prepare_migration" },
        api::resharding::prepare_migration(),
        transformer(argument<gid_t>("gid"), resharding_role_arg));
    bind.plain(
        { "/resharding/xtable/start_migration" },
        api::resharding::start_migration(),
        transformer(argument<gid_t>("gid"), resharding_role_arg, request_reaction_arg));
    bind.plain(
        { "/resharding/xtable/finalize_migration" },
        api::resharding::finalize_migration(),
        transformer(argument<gid_t>("gid"), resharding_role_arg));
    bind.plain(
        { "/resharding/xtable/abort_migration" },
        api::resharding::abort_migration(),
        transformer(argument<gid_t>("gid"), resharding_role_arg));
    bind.plain({ "/resharding/xtable/status" }, api::resharding::status());

    // use custom key extractor while webserver doesnt support paths with wildcards
    webserver_module->set_custom_key_extractor(
        "front", [](ymod_webserver::http::stream_ptr stream) -> string {
            auto result = stream->request()->url.make_full_path();

            static const std::vector<string> patterns = { "/notify/",
                                                          "/fast_notify/",
                                                          "/send/",
                                                          "/binary_notify/",
                                                          "/fast_binary_notify/",
                                                          "/enable/",
                                                          "/disable/",
                                                          "/batch_binary_notify/",
                                                          "/batch_binary_notify_single/",
                                                          "/batch_binary_notify_small/",
                                                          "/batch_binary_notify_medium/" };

            for (auto& pattern : patterns)
            {
                if (starts_with(result, pattern))
                {
                    result = pattern + "*";
                    break;
                }
            }
            return result;
        });
}

}}
