#pragma once

#include "common.h"
#include "../control_module.h"

namespace yxiva { namespace hub { namespace api {

struct degrade_control : public api_coroutine
{
    std::shared_ptr<state> hub;
    ymod_webserver::http::stream_ptr stream;

    void operator()()
    {
        auto& path = stream->request()->url.path;
        if (path.size() < 2 || path[1].empty())
        {
            stream->result(
                ymod_webserver::codes::bad_request, "invalid path, must be /[en|dis]able/what");
            return;
        }

        bool enable = path[0][0] == 'e';
        auto what = path[1][0];
        switch (what)
        {
        case 'a':
            enable ? enable_all() : disable_all();
            break;
        case 't':
            enable ? enable_tasks() : disable_tasks();
            break;
        case 's':
            enable ? enable_store() : disable_store();
            break;
        case 'x':
            enable ? enable_xtable() : disable_xtable();
            break;
        case 'd':
            enable ? enable_deduplication() : disable_deduplication();
            break;
        case 'w':
            enable ? enable_worker() : disable_worker();
            break;
        case 'p':
            enable ? enable_progressive_retry() : disable_progressive_retry();
            break;
        case 'r':
            enable ? enable_replica_reads() : disable_replica_reads();
            break;
        case 'c':
            enable ? enable_cluster_pq_auto_fallback() : disable_cluster_pq_auto_fallback();
            break;
        default:
            stream->result(
                ymod_webserver::codes::bad_request, "invalid system code: " + string(1, what));
            break;
        }
    }

    void enable_all()
    {
        hub->stats.ignore_xstore_errors = false;
        hub->stats.deduplicate_xtable = true;
        hub->xtasks_worker->toggle_progressive_retry(true);
        hub->xtasks_worker->start();
        hub->xtable->force_db_role({});
        hub->pq_xtable->enable_auto_fallback(stream->ctx());
        hub->pq_xstore->enable_auto_fallback(stream->ctx());
        hub->pq_batch->enable_auto_fallback(stream->ctx());
        hub->stats.manual_xtasks_degradation = false;
        hub->stats.manual_xstore_degradation = false;
        set_robust_delivery(true);
        stream->result(codes::ok);
    }

    void disable_all()
    {
        stream->result(
            ymod_webserver::codes::bad_request, "cannot disable all systems in one call");
    }

    void enable_tasks()
    {
        hub->stats.manual_xtasks_degradation = false;
        stream->result(set_robust_delivery(true) ? codes::ok : codes::forbidden);
    }

    void disable_tasks()
    {
        hub->stats.manual_xtasks_degradation = true;
        stream->result(set_robust_delivery(false) ? codes::ok : codes::forbidden);
    }

    void enable_store()
    {
        hub->xstore->disable_fallback(stream->ctx());
        hub->stats.manual_xstore_degradation = false;
        hub->stats.ignore_xstore_errors = false;
        stream->result(ymod_webserver::codes::ok, "ok");
    }

    void disable_store()
    {
        if (hub->stats.robust_delivery)
        {
            stream->result(
                ymod_webserver::codes::forbidden,
                "can't degrade xstore without degrading xtasks first");
            return;
        }
        hub->stats.manual_xstore_degradation = true;
        hub->stats.ignore_xstore_errors = true;
        hub->xstore->enable_fallback(stream->ctx());
        stream->result(ymod_webserver::codes::ok, "ok");
    }

    void enable_xtable()
    {
        hub->xtable->disable_fallback(stream->ctx());
        stream->result(ymod_webserver::codes::ok, "ok");
    }

    void disable_xtable()
    {
        hub->xtable->enable_fallback(stream->ctx());
        stream->result(ymod_webserver::codes::ok, "ok");
    }

    void enable_deduplication()
    {
        hub->stats.deduplicate_xtable = true;
        stream->result(ymod_webserver::codes::ok, "ok");
    }

    void disable_deduplication()
    {
        hub->stats.deduplicate_xtable = false;
        stream->result(ymod_webserver::codes::ok, "ok");
    }

    void enable_worker()
    {
        hub->xtasks_worker->start();
        stream->result(ymod_webserver::codes::ok, "ok");
    }

    void disable_worker()
    {
        hub->xtasks_worker->stop();
        stream->result(ymod_webserver::codes::ok, "ok");
    }

    void enable_progressive_retry()
    {
        hub->xtasks_worker->toggle_progressive_retry(true);
        stream->result(ymod_webserver::codes::ok, "ok");
    }

    void disable_progressive_retry()
    {
        hub->xtasks_worker->toggle_progressive_retry(false);
        stream->result(ymod_webserver::codes::ok, "ok");
    }

    void disable_replica_reads()
    {
        hub->xtable->force_db_role(XTable::db_role::master);
        stream->result(ymod_webserver::codes::ok, "ok");
    }

    void enable_replica_reads()
    {
        hub->xtable->force_db_role({});
        stream->result(ymod_webserver::codes::ok, "ok");
    }

    void enable_cluster_pq_auto_fallback()
    {
        hub->pq_xtable->enable_auto_fallback(stream->ctx());
        hub->pq_xstore->enable_auto_fallback(stream->ctx());
        hub->pq_batch->enable_auto_fallback(stream->ctx());
        stream->result(ymod_webserver::codes::ok, "ok");
    }

    void disable_cluster_pq_auto_fallback()
    {
        hub->pq_xtable->disable_auto_fallback(stream->ctx());
        hub->pq_xstore->disable_auto_fallback(stream->ctx());
        hub->pq_batch->disable_auto_fallback(stream->ctx());
        stream->result(ymod_webserver::codes::ok, "ok");
    }

    bool set_robust_delivery(bool robust)
    {
        auto controller = hub->controller.lock();
        return controller && controller->async_set_robust_delivery(robust);
    }
};
}}}
