#pragma once

#include "common.h"
#include <yxiva/core/subscriptions.h>

namespace yxiva { namespace hub { namespace api {

void send_list_json(ymod_webserver::http::stream_ptr stream, const sub_list& list);

struct list_json : public api_coroutine
{
    std::shared_ptr<state> hub;
    ymod_webserver::http::stream_ptr stream;
    string uid;
    string service;
    XTable::db_role db_role;

    auto find_options()
    {
        XTable::find_options options;
        options.show_inactive = true;
        options.db_role = db_role;
        return options;
    }

    void operator()(const error_code& err = {}, const sub_list& list = {})
    {
        reenter(*this)
        {
            yield hub->xtable->find(stream->ctx(), uid, service, find_options(), *this);
            if (err)
            {
                stream->result(http_code_for_error(err), message_for_error(err));
                return;
            }
            send_list_json(stream, list);
        }
    }
};

struct list_json_uidset : public api_coroutine
{
    std::shared_ptr<state> hub;
    ymod_webserver::http::stream_ptr stream;
    string uidset;
    string service;

    auto find_options()
    {
        XTable::find_options options;
        options.show_inactive = true;
        options.db_role = XTable::db_role::replica;
        return options;
    }

    void operator()(const error_code& err = {}, const sub_list& list = {})
    {
        reenter(this)
        {
            yield hub->xtable->find_uidset(stream->ctx(), uidset, service, find_options(), *this);
            if (err)
            {
                stream->result(http_code_for_error(err), message_for_error(err));
                return;
            }
            send_list_json(stream, list);
        }
    }
};

struct batch_list_json : public api_coroutine
{
    std::shared_ptr<state> hub;
    ymod_webserver::http::stream_ptr stream;
    string uid;
    std::vector<string> services;

    auto find_options()
    {
        XTable::find_options options;
        options.show_inactive = true;
        options.db_role = XTable::db_role::replica;
        return options;
    }

    void operator()(const error_code& err = {}, const sub_list& list = {})
    {
        reenter(this)
        {
            yield hub->xtable->batch_find(stream->ctx(), uid, services, find_options(), *this);
            if (err)
            {
                stream->result(http_code_for_error(err), message_for_error(err));
                return;
            }
            send_list_json(stream, list);
        }
    }
};

struct batch_uids_list_json : public api_coroutine
{
    std::shared_ptr<state> hub;
    ymod_webserver::http::stream_ptr stream;
    std::vector<string> uids;
    string service;

    // Share intermediate results and keys
    // because batch_find() may call handler many times.
    struct data
    {
        batch_keys keys;
        size_t remaining_keys = 0;
        sub_list list;
        error_code err;
    };
    std::shared_ptr<data> shared{ new data };

    void operator()(
        const error_code& err = {},
        const batch_iterators& batch = {},
        const sub_list& list = {})
    {
        reenter(this)
        {
            shared->remaining_keys = uids.size();
            shared->keys.reserve(uids.size());
            for (auto&& uid : uids)
            {
                shared->keys.push_back(batch_key{ uid, {} });
            }
            yield hub->xtable->batch_find(
                stream->ctx(), shared->keys, service, {}, stream->get_io_service().wrap(*this));
            shared->remaining_keys -= batch.size();
            accumulate_batch_result(err, list);
            if (shared->remaining_keys) return;
            send_result();
        }
    }

    void accumulate_batch_result(const error_code& err, const sub_list& list)
    {
        if (err)
        {
            // Last error is sufficient for making an adequate reply.
            shared->err = err;
        }
        else
        {
            shared->list.insert(shared->list.end(), list.begin(), list.end());
        }
    }

    void send_result()
    {
        if (shared->err)
        {
            stream->result(http_code_for_error(shared->err), message_for_error(shared->err));
        }
        else
        {
            send_list_json(stream, shared->list);
        }
    }
};

}}}
