#pragma once

#include "common.h"
#include "hashing.h"
#include "batch_notify_coro.h"
#include <yxiva/core/services/csv_decoder.h>
#include <yxiva/core/services/simple_http_decoder.h>
#include <yplatform/util/split.h>
#include <boost/lexical_cast.hpp>

namespace yxiva { namespace hub { namespace api {

inline auto create_predefined_decoders()
{
    auto decoders = boost::make_shared<services::decoders_kit>();
    (*decoders)["mail"] = boost::make_shared<services::csv_decoder>("mail");
    (*decoders)["altmail"] = boost::make_shared<services::csv_decoder>("altmail");
    (*decoders)[""] = boost::make_shared<services::simple_http_decoder>("");
    return decoders;
}

struct notify : public api_coroutine
{
    std::shared_ptr<state> hub;
    expirable_stream_ptr stream;
    bool binary;
    bool fast;
    std::vector<string> tags;
    std::time_t event_ts;
    string message_topic;
    bool enable_deduplication;

    ymod_webserver::request_ptr req = {};
    string service = {};
    shared_ptr<message> msg = { make_shared<message>() };
    string message_hash = {};

    void operator()(const error_code& ec = {}, const string& convey_status = {})
    {
        (void)convey_status;
        bool weak_delivery = false;

        reenter(this)
        {
            req = req = stream->request();
            service = extract_service_name(*stream->request());

            if (binary)
            {
                // For now, allow empty service.
                try
                {
                    unpack(string(req->raw_body.begin(), req->raw_body.end()), *msg);
                }
                catch (const std::exception& e)
                {
                    WEB_RESPONSE_LOG_G(info, stream, bad_request, "failed to unpack body");
                    return;
                }
                if (service.size() && msg->service != service)
                {
                    WEB_RESPONSE_LOG_G(
                        error,
                        stream,
                        bad_request,
                        "service name from message and from path do not match");
                    return;
                }
                setup_ts(*msg, event_ts);
            }
            else
            {
                string uid = stream->request()->url.param_value("uid", "");
                if (service.empty())
                {
                    WEB_RESPONSE_LOG_G(error, stream, bad_request, "missing service name");
                    return;
                }

                // decode message
                static auto decoders = create_predefined_decoders();
                auto it = decoders->find(service);
                // get the default decoder if no specific decoder found
                if (it == decoders->end()) it = decoders->find(service_name(""));
                if (it != decoders->end())
                {
                    it->second->decode(stream->request(), *msg);
                }
                else
                {
                    WEB_RESPONSE_LOG_G(
                        error,
                        stream,
                        bad_request,
                        "no decoder for " + static_cast<string>(service));
                    return;
                }
                msg->service = service;

                if (tags.size() >= hub->settings->message_max_tags)
                {
                    WEB_RESPONSE_LOG_G(info, stream, bad_request, "too many tags");
                    return;
                }
                msg->tags = tags;
                msg->transit_id = stream->ctx()->uniq_id();
                setup_ts(*msg, event_ts);
                msg->uid = uid;
            }

            message_hash = hash(*msg);
            hub->transport_log->new_notification(stream->ctx(), *msg, false, message_hash);

            if (fast)
            {
                if (stream->request()->url.param_value("report", "0") == "1")
                {
                    batch_key key;
                    key.uid = msg->uid;
                    batch_notify_coro coro = { {},
                                               hub,
                                               stream,
                                               msg,
                                               batch_keys{ std::move(key) },
                                               hub->settings->send_result_timeout_delta,
                                               {} };
                    coro();
                }
                else
                {
                    yield hub->exhaust->convey(stream->ctx(), msg, *this);
                    if (ec && ec.value() != err_code_subscription_dropped)
                    {
                        WEB_RESPONSE_LOG_G(
                            error, stream, internal_server_error, "xtable list failed");
                    }
                    else
                    {
                        reply_ok(stream, msg->local_id, msg->transit_id);
                    }
                }
                return;
            }

            if (!hub->stats.convey_enabled)
            {
                hub->transport_log->notification_convey_disabled(stream->ctx(), *msg);
                WEB_RESPONSE(stream, internal_server_error, "convey disabled");
                return;
            }

            // write to xstore
            stream->context()->profilers.push("xstore::write");
            yield
            {
                if (!enable_deduplication) message_hash += msg->transit_id;
                ymod_xstore::entry xstore_entry;
                xstore_entry.uid = msg->uid;
                xstore_entry.service = msg->service;
                xstore_entry.event_ts = msg->event_ts;
                xstore_entry.ttl = msg->ttl;
                xstore_entry.content = pack(*msg);
                xstore_entry.transit_id = msg->transit_id;
                xstore_entry.topic = message_topic;
                hub->xstore->write(stream->context(), std::move(xstore_entry), message_hash, *this);
            }
            stream->context()->profilers.pop("xstore::write");

            if (stream->context()->is_cancelled()) return;

            if (ec)
            {
                if (!hub->stats.ignore_xstore_errors)
                {
                    hub->transport_log->notification_save_failed(stream->ctx(), *msg, ec.message());
                    WEB_RESPONSE_LOG_G(error, stream, internal_server_error, "xstore write error");
                    return;
                }
                else
                {
                    YLOG_CTX_GLOBAL(stream->ctx(), warning) << "ignoring xstore error";
                }
            }

            weak_delivery = !hub->stats.robust_delivery || (ec && hub->stats.ignore_xstore_errors);
            if (weak_delivery)
            {
                if (stream->request()->url.param_value("report", "0") == "1")
                {
                    batch_key key;
                    key.uid = msg->uid;
                    batch_notify_coro coro = { {},
                                               hub,
                                               stream,
                                               msg,
                                               batch_keys{ std::move(key) },
                                               hub->settings->send_result_timeout_delta,
                                               {} };
                    coro();
                }
                else
                {
                    yield hub->exhaust->convey(stream->ctx(), msg, *this);
                    if (ec && ec.value() != err_code_subscription_dropped)
                    {
                        WEB_RESPONSE_LOG_G(
                            error, stream, internal_server_error, "xtable list failed");
                    }
                    else
                    {
                        reply_ok(stream, msg->local_id, msg->transit_id);
                    }
                }
                return;
            }

            stream->context()->profilers.push("xtasks::create");
            yield
            {
                ymod_xtasks::task_draft draft = {
                    msg->uid, msg->service, msg->local_id, "", ymod_xtasks::delay_flags::none
                };
                hub->xtasks->create_task(stream->context(), draft, *this);
            }
            stream->context()->profilers.pop("xtasks::create");

            if (ec)
            {
                hub->transport_log->notification_task_create_failed(
                    stream->ctx(), *msg, ec.message());

                // collect errors, observer timer handler will analyse it later
                if (hub->stats.control_leader)
                {
                    hub->stats.xtasks_errors++;
                }

                WEB_RESPONSE(stream, internal_server_error, "xtask create error");
                return;
            }

            hub->transport_log->notification_task_created(stream->ctx(), *msg);

            reply_ok(stream, msg->local_id, msg->transit_id);
        }
    }

    // xstore.write handler
    void operator()(
        const ymod_xstore::error& error,
        const local_id_t local_id,
        const std::time_t event_ts,
        ymod_xstore::merge_type merge,
        const string& transit_id)
    {
        msg->local_id = local_id;
        msg->event_ts = event_ts;
        string old_transit_id;
        if (merge != ymod_xstore::merge_type::none && !transit_id.empty())
        {
            old_transit_id = msg->transit_id;
            msg->transit_id = transit_id;
        }
        // TODO move to coroutine body
        if (merge == ymod_xstore::merge_type::duplicate)
        {
            reply_ok(stream, msg->local_id, msg->transit_id);
            return;
        }
        (*this)(error.code);
    }

    // xtasks.create handler
    void operator()(const ymod_xtasks::error& error)
    {
        (*this)(error.code);
    }

    void reply_ok(expirable_stream_ptr stream, local_id_t local_id, const string& transit_id)
    {
        if (auto http_stream = stream->detach_stream())
        {
            http_stream->set_code(ymod_webserver::codes::ok);
            http_stream->add_header("NotificationID", std::to_string(local_id));
            http_stream->add_header("TransitID", transit_id);
            http_stream->result_body("");
        }
    }

    string extract_service_name(ymod_webserver::request& req)
    {
        return req.url.path.size() > 1 ? req.url.path[1] : string();
    }

    void setup_ts(message& msg, std::time_t desirable_value)
    {
        // Internally specified value is preferable.
        if (!msg.event_ts)
        {
            msg.event_ts = desirable_value ? desirable_value : std::time(nullptr);
        }
        // Time in ms can't be specified externally.
        if (!msg.event_ts_ms)
        {
            msg.event_ts_ms = time_ms();
        }
    }
};

struct simple_notify
{
    std::shared_ptr<state> hub;
    expirable_stream_ptr stream;
    std::vector<string> tags;
    std::time_t event_ts;
    string message_topic;
    bool enable_deduplication;

    void operator()()
    {
        notify impl = { {},
                        hub,
                        stream,
                        false,
                        false,
                        std::move(tags),
                        event_ts,
                        std::move(message_topic),
                        enable_deduplication };
        impl();
    }
};

struct fast_notify
{
    std::shared_ptr<state> hub;
    expirable_stream_ptr stream;
    std::vector<string> tags;
    std::time_t event_ts;
    string message_topic;
    bool enable_deduplication;

    void operator()()
    {
        notify impl = { {},
                        hub,
                        stream,
                        false,
                        true,
                        std::move(tags),
                        event_ts,
                        std::move(message_topic),
                        enable_deduplication };
        impl();
    }
};

struct binary_notify
{
    std::shared_ptr<state> hub;
    expirable_stream_ptr stream;
    std::vector<string> tags;
    std::time_t event_ts;
    string message_topic;
    bool enable_deduplication;

    void operator()()
    {
        notify impl = { {},
                        hub,
                        stream,
                        true,
                        false,
                        std::move(tags),
                        event_ts,
                        std::move(message_topic),
                        enable_deduplication };
        impl();
    }
};

struct fast_binary_notify
{
    std::shared_ptr<state> hub;
    expirable_stream_ptr stream;
    std::vector<string> tags;
    std::time_t event_ts;
    string message_topic;
    bool enable_deduplication;

    void operator()()
    {
        notify impl = { {},
                        hub,
                        stream,
                        true,
                        true,
                        std::move(tags),
                        event_ts,
                        std::move(message_topic),
                        enable_deduplication };
        impl();
    }
};

}}}
