#pragma once

#include <api/common.h>
#include <resharding/controller.h>
#include <yxiva/core/shards/merging_storage.h>

namespace yxiva { namespace hub { namespace api { namespace resharding {

using controller = hub::resharding::controller;
using migration_state = yxiva::resharding::migration::state_type;

controller::request_reaction request_reaction_from_string(const string& s);
controller::role role_from_string(const string& s);

namespace detail {
class dummy_controller : public controller
{
public:
    static const boost::shared_ptr<dummy_controller>& instance()
    {
        static const auto the_dummy = boost::make_shared<dummy_controller>();
        return the_dummy;
    }

    virtual void prepare_migration(
        task_context_ptr,
        gid_t,
        role,
        role,
        const std::string&,
        const set_state_callback& cb) override
    {
        cb(make_error_code(err_code_resharding_disabled), {});
    }

    virtual void start_migration(
        task_context_ptr,
        gid_t,
        role,
        role,
        const std::string&,
        request_reaction,
        const set_state_callback& cb) override
    {
        cb(make_error_code(err_code_resharding_disabled), {});
    }

    virtual void finalize_migration(
        task_context_ptr,
        gid_t,
        role,
        role,
        const std::string&,
        const set_state_callback& cb) override
    {
        cb(make_error_code(err_code_resharding_disabled), {});
    }

    virtual void abort_migration(
        task_context_ptr,
        gid_t,
        role,
        role,
        const std::string&,
        const set_state_callback& cb) override
    {
        cb(make_error_code(err_code_resharding_disabled), {});
    }

    virtual void list_migrations(task_context_ptr, const list_migrations_callback& cb) override
    {
        cb(make_error_code(err_code_resharding_disabled), {});
    }
};

inline auto controller(const boost::shared_ptr<XTable>& xtable)
{
    auto controller = boost::dynamic_pointer_cast<hub::resharding::controller>(xtable);
    return controller ? controller : dummy_controller::instance();
}

inline auto role(const std::shared_ptr<state>& hub)
{
    auto& stats = hub->stats;
    if (!stats.convey_enabled || !stats.robust_delivery)
    {
        return controller::role::unknown;
    }
    return stats.control_leader ? controller::role::master : controller::role::slave;
}

inline auto master(const std::shared_ptr<state>& hub)
{
    return hub->stats.get_cluster_metadata().master_address;
}

inline void handle_controller_response(
    const stream_ptr& stream,
    const error_code& ec,
    migration_state new_state)
{
    if (ec && ec.value() != err_code_resharding_bad_migration_state)
    {
        WEB_RESPONSE_CODE_LOG_G(error, stream, http_code_for_error(ec), message_for_error(ec));
    }
    else
    {
        WEB_RESPONSE_CODE(
            stream, http_code_for_error(ec), "{ \"new_state\": \"" + to_string(new_state) + "\" }");
    }
}
}

struct prepare_migration : public api_coroutine
{
    std::shared_ptr<state> hub;
    stream_ptr stream;
    gid_t gid;
    controller::role expected_role;

    void operator()(const error_code& ec = {}, migration_state new_state = {})
    {
        reenter(this)
        {
            yield detail::controller(hub->xtable)
                ->prepare_migration(
                    stream->ctx(),
                    gid,
                    expected_role,
                    detail::role(hub),
                    detail::master(hub),
                    *this);
            detail::handle_controller_response(stream, ec, new_state);
        }
    }
};

struct start_migration : public api_coroutine
{
    std::shared_ptr<state> hub;
    stream_ptr stream;
    gid_t gid;
    controller::role expected_role;
    controller::request_reaction reaction;

    void operator()(const error_code& ec = {}, migration_state new_state = {})
    {
        reenter(this)
        {
            yield detail::controller(hub->xtable)
                ->start_migration(
                    stream->ctx(),
                    gid,
                    expected_role,
                    detail::role(hub),
                    detail::master(hub),
                    reaction,
                    *this);
            detail::handle_controller_response(stream, ec, new_state);
        }
    }
};

struct finalize_migration : public api_coroutine
{
    std::shared_ptr<state> hub;
    stream_ptr stream;
    gid_t gid;
    controller::role expected_role;

    void operator()(const error_code& ec = {}, migration_state new_state = {})
    {
        reenter(this)
        {
            yield detail::controller(hub->xtable)
                ->finalize_migration(
                    stream->ctx(),
                    gid,
                    expected_role,
                    detail::role(hub),
                    detail::master(hub),
                    *this);
            detail::handle_controller_response(stream, ec, new_state);
        }
    }
};

struct abort_migration : public api_coroutine
{
    std::shared_ptr<state> hub;
    stream_ptr stream;
    gid_t gid;
    controller::role expected_role;

    void operator()(const error_code& ec = {}, migration_state new_state = {})
    {
        reenter(this)
        {
            yield detail::controller(hub->xtable)
                ->abort_migration(
                    stream->ctx(),
                    gid,
                    expected_role,
                    detail::role(hub),
                    detail::master(hub),
                    *this);
            detail::handle_controller_response(stream, ec, new_state);
        }
    }
};

struct status : public api_coroutine
{
    std::shared_ptr<state> hub;
    stream_ptr stream;

    void operator()(const error_code& ec = {}, const yxiva::resharding::migrations& migrations = {})
    {
        reenter(this)
        {
            yield detail::controller(hub->xtable)->list_migrations(stream->ctx(), *this);
            if (ec)
            {
                WEB_RESPONSE_CODE_LOG_G(
                    error, stream, http_code_for_error(ec), message_for_error(ec));
                return;
            }

            auto merged_shards =
                std::dynamic_pointer_cast<shard_config::merging_storage>(hub->xtable->shards());
            if (!merged_shards)
            {
                WEB_RESPONSE_CODE_LOG_G(
                    error,
                    stream,
                    ymod_webserver::codes::internal_server_error,
                    "xtable is not configured for resharding");
                return;
            }

            auto old_shards = merged_shards->shards_from();
            auto new_shards = merged_shards->shards_to();

            if (!old_shards || !new_shards)
            {
                WEB_RESPONSE_CODE_LOG_G(
                    error,
                    stream,
                    ymod_webserver::codes::internal_server_error,
                    "no shards_from or shards_to available");
                return;
            }

            json_value result;
            result["old_shards"] = to_json(*old_shards->get());
            result["new_shards"] = to_json(*new_shards->get());
            result["migrations"] = to_json(migrations);
            stream->result(ymod_webserver::codes::ok, json_write_styled(result));
        }
    }
};

}}}}
